/*
 * Decompiled with CFR 0.152.
 */
package com.stratadata.model3.well.curve;

import com.stratadata.model3.audit.Audit;
import com.stratadata.model3.audit.AuditImpl;
import com.stratadata.model3.user.UserService;
import com.stratadata.model3.well.curve.Curve;
import com.stratadata.model3.well.curve.WellCurveService;
import java.io.BufferedWriter;
import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import model3.SBdb;
import model3.Well;
import model3.exception.SuppressedSQLException;
import org.jdom2.Element;
import util.SB;
import util.SBException;
import util.SBPermissionException;
import util.listener.WeakListenerList;

public class Curves
implements WellCurveService {
    private static final Logger LOGGER = Logger.getLogger(Curves.class.getName());
    private final SBdb db;
    private final int wellID;
    private final Map<Integer, Curve> curves = new HashMap<Integer, Curve>();
    private final WeakListenerList<WellCurveService.Listener> listenerList = new WeakListenerList();
    private boolean hasChanged = false;

    public Curves(SBdb db, int wellID) throws SQLException {
        this.db = db;
        this.wellID = wellID;
        this.load();
    }

    private void load() throws SQLException {
        if (this.db.isConnected()) {
            Logger.getLogger(this.getClass().getName()).log(Level.CONFIG, "Loading wireline log curves for well: " + this.wellID);
            String sql = "SELECT curve_id, abr, curve_mnem, filename, comments, " + model3.Audit.sqlFieldString() + " FROM " + this.db.DBTableName("LOG_CURVE") + " WHERE well_id=" + this.wellID;
            String traceSql = "SELECT depth, trace_value FROM " + this.db.DBTableName("LOG_TRACE") + " WHERE curve_id=? ORDER BY depth";
            try (Statement stmt = this.db.getDatabase().createStatement();
                 PreparedStatement pStmt = this.db.getDatabase().prepareStatement(this.db.modQuery(traceSql));){
                LOGGER.log(Level.FINE, "Curve statement fetch size: " + stmt.getFetchSize());
                pStmt.setFetchSize(10000);
                LOGGER.log(Level.FINE, "Trace statement fetch size: " + pStmt.getFetchSize());
                ResultSet rs = stmt.executeQuery(this.db.modQuery(sql));
                while (rs.next()) {
                    int curveID = rs.getInt("curve_id");
                    String abr = rs.getString("abr");
                    String curveMnem = rs.getString("curve_mnem");
                    String fileName = rs.getString("filename");
                    String comments = rs.getString("comments");
                    AuditImpl audit = model3.Audit.getAuditFromResultSet(rs);
                    LinkedList<Curve.CurveValue> trace = new LinkedList<Curve.CurveValue>();
                    pStmt.setInt(1, curveID);
                    ResultSet rs2 = pStmt.executeQuery();
                    while (rs2.next()) {
                        double depth = rs2.getDouble("depth");
                        double value = rs2.getDouble("trace_value");
                        trace.add(new Curve.CurveValue(depth, value));
                    }
                    this.curves.put(curveID, new Curve(curveID, trace, abr, curveMnem, fileName, comments, audit));
                }
            }
        }
    }

    public List<Curve> getCurves(boolean activeOnly) {
        return this.curves.values().stream().filter(activeOnly ? Curve::isActive : curve -> true).collect(Collectors.toCollection(ArrayList::new));
    }

    public List<Curve> getCurves(String abr) {
        return this.curves.values().stream().filter(curve -> curve.isActive() && curve.getAbr().equals(abr)).toList();
    }

    public int getCurveCount() {
        return (int)this.curves.values().stream().filter(Curve::isActive).count();
    }

    public void add(Curve curve) throws SQLException {
        if (this.db.isConnected()) {
            curve = this.store(curve);
        } else assert (curve.getID() > 0);
        this.curves.put(curve.getID(), curve);
        this.setChanged();
    }

    private Curve store(Curve curve) throws SQLException {
        Curve storedCurve;
        int curveID = this.db.nextControl("LOG_CURVE", "CURVE_ID");
        try (Statement stmt = this.db.getDatabase().createStatement();){
            AuditImpl audit = AuditImpl.getAuditUpdate((Audit)curve.getAudit(), (UserService)this.db.getUserService(), (Instant)model3.Audit.getDatabaseServerDate(this.db, stmt).toInstant(), (boolean)false);
            String sql = "INSERT INTO " + this.db.DBTableName("LOG_CURVE") + " (curve_id,well_id,abr,curve_mnem,filename,comments," + model3.Audit.sqlFieldString() + ") VALUES (" + curveID + "," + this.wellID + "," + SB.DBString((String)curve.getAbr()) + "," + SB.DBString((String)curve.getMnem()) + "," + SB.DBString((String)curve.getFilename()) + "," + SB.DBString((String)curve.getComments()) + "," + model3.Audit.sqlInsert((Audit)audit) + ")";
            stmt.executeUpdate(this.db.modQuery(sql));
            this.db.commit();
            storedCurve = Curve.copyOf((Curve)curve, (int)curveID, (AuditImpl)audit, (String)curve.getAbr());
        }
        int[] affectedRows = this.storeTrace(curveID, storedCurve.getTrace());
        int nInserted = 0;
        int nError = 0;
        int nNoInfo = 0;
        for (int row : affectedRows) {
            if (row > 0) {
                ++nInserted;
                continue;
            }
            if (row == -3) {
                ++nError;
                continue;
            }
            ++nNoInfo;
        }
        LOGGER.log(Level.CONFIG, "Number of rows affected:" + affectedRows.length + ", Inserted:" + nInserted + ", Error:" + nError + ", NoInfo:" + nNoInfo);
        if (nError == affectedRows.length) {
            throw new RuntimeException("Database error: no log_curve values saved");
        }
        return storedCurve;
    }

    private int[] storeTrace(int curveID, List<Curve.CurveValue> trace) throws SQLException {
        String sql = "INSERT INTO " + this.db.DBTableName("LOG_TRACE") + " (curve_id, depth, trace_value) VALUES (" + curveID + ",?,?)";
        try (PreparedStatement pStmt = this.db.getDatabase().prepareStatement(this.db.modQuery(sql));){
            for (Curve.CurveValue value : trace) {
                pStmt.setDouble(1, value.depth());
                pStmt.setDouble(2, value.value());
                pStmt.addBatch();
            }
            int[] nArray = pStmt.executeBatch();
            return nArray;
        }
    }

    public void deleteCurves(Collection<Curve> toDelete) throws SuppressedSQLException {
        if (toDelete.isEmpty()) {
            return;
        }
        try {
            Well well = this.db.getWell(this.wellID);
            if (!well.canWrite(this.db, null)) {
                throw new SBPermissionException(well.getDeniedReason(this.db, "well", true));
            }
        }
        catch (SQLException | SBException | SBPermissionException e) {
            throw new RuntimeException(e);
        }
        if (this.db.isConnected()) {
            try (Statement stmt = this.db.getDatabase().createStatement();){
                for (Curve curve : toDelete) {
                    if (this.curves.get(curve.getID()) == null) {
                        throw new IllegalArgumentException("Attempt to delete curve from wrong well");
                    }
                    String sql = "DELETE FROM " + this.db.DBTableName("LOG_TRACE") + " WHERE curve_id=" + curve.getID();
                    stmt.executeUpdate(this.db.modQuery(sql));
                    sql = "DELETE FROM " + this.db.DBTableName("LOG_CURVE") + " WHERE curve_id=" + curve.getID();
                    stmt.executeUpdate(this.db.modQuery(sql));
                }
                this.db.commit();
            }
            catch (SQLException sqlException) {
                throw SuppressedSQLException.withRollback("Error deleting curve", sqlException, this.db);
            }
        }
        for (Curve curve : toDelete) {
            this.curves.remove(curve.getID());
        }
        if (this.db.isConnected()) {
            try {
                for (Curve curve : toDelete) {
                    this.db.updateAuditTrail("LOG_CURVE", "DELETE " + curve.getAbr() + " from " + String.valueOf(this) + " [" + this.wellID + "]");
                }
            }
            catch (SQLException sqlException) {
                throw SuppressedSQLException.withRollback("Error updating audit trail", sqlException, this.db);
            }
        }
        this.setChanged();
        this.notifyListeners();
    }

    public Curve updateCurve(Curve updatedCurve) throws SQLException {
        AuditImpl updatedAudit;
        int curveID = updatedCurve.getID();
        if (curveID == 0) {
            throw new IllegalStateException("Attempt to update curve with ID:0");
        }
        Curve existingCurve = this.getCurves().stream().filter(c -> c.getID() == curveID).findFirst().orElseThrow();
        try (Statement stmt = this.db.getDatabase().createStatement();){
            updatedAudit = AuditImpl.getAuditUpdate((Audit)existingCurve.getAudit(), (UserService)this.db.getUserService(), (Instant)model3.Audit.getDatabaseServerDate(this.db, stmt).toInstant(), (boolean)false);
            String sql = "DELETE FROM " + this.db.DBTableName("LOG_TRACE") + " WHERE curve_id=" + curveID;
            stmt.executeUpdate(this.db.modQuery(sql));
            sql = "UPDATE " + this.db.DBTableName("LOG_CURVE") + " set comments=" + SB.DBString((String)updatedCurve.getComments()) + ", " + model3.Audit.sqlUpdate((Audit)updatedAudit) + " WHERE curve_id=" + curveID;
            stmt.executeUpdate(this.db.modQuery(sql));
            this.storeTrace(curveID, updatedCurve.getTrace());
            this.db.commit();
        }
        this.curves.remove(curveID);
        Curve newCurve = Curve.copyOf((Curve)updatedCurve, (int)curveID, (AuditImpl)updatedAudit, (String)updatedCurve.getAbr());
        this.curves.put(updatedCurve.getID(), newCurve);
        this.setChanged();
        this.notifyListeners();
        return newCurve;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void refresh(SBdb db, Statement stmt, int wellID) throws SQLException, SBException {
        Object sql = "SELECT curve_id,updated ";
        sql = (String)sql + " FROM " + db.DBTableName("LOG_CURVE") + " WHERE well_id=" + wellID;
        sql = db.modQuery((String)sql);
        ResultSet rs = stmt.executeQuery((String)sql);
        Curve notifier = null;
        HashSet<Integer> keys = new HashSet<Integer>();
        boolean reload = false;
        Map<Integer, Curve> map = this.curves;
        synchronized (map) {
            while (rs.next()) {
                int key = rs.getInt("curve_id");
                keys.add(key);
                Timestamp time = rs.getTimestamp("updated");
                boolean found = false;
                for (Curve o : this.curves.values()) {
                    if (o.getID() != key) continue;
                    found = true;
                    if (time == null || !o.getAudit().getUpdated().isBefore(((Date)time).toInstant())) break;
                    reload = true;
                    notifier = o;
                    break;
                }
                if (found) continue;
                reload = true;
                break;
            }
        }
        if (reload || keys.size() < this.curves.size()) {
            if (notifier == null) {
                map = this.curves;
                synchronized (map) {
                    notifier = this.curves.get(0);
                }
            }
            this.curves.clear();
            this.load();
            this.setChanged();
            if (notifier == null) {
                map = this.curves;
                synchronized (map) {
                    notifier = this.curves.get(0);
                }
            }
            this.notifyListeners();
        }
    }

    public void writeXML(BufferedWriter out, int indent) throws IOException {
        for (Curve curve : this.curves.values()) {
            String ind1 = SB.getXMLIndent((int)indent);
            out.write(ind1 + "<WirelineLog LogDef=\"" + curve.getAbr() + "\">\n");
            String ind = SB.getXMLIndent((int)(indent + 3));
            out.write(ind + "<CurveID>" + curve.getID() + "</CurveID>\n");
            out.write(ind + "<Mnemonic>" + SB.getXMLstring((String)curve.getMnem()) + "</Mnemonic>\n");
            if (curve.getComments() != null && !curve.getComments().isEmpty()) {
                out.write(ind + "<Comments>" + SB.getXMLstring((String)curve.getComments()) + "</Comments>\n");
            }
            String rowString = ind + "<Row Depth=\"%s\" Value=\"%s\"/>\n";
            for (Curve.CurveValue row : curve.getTrace()) {
                out.write(String.format(rowString, row.depth(), row.value()));
            }
            out.write(ind1 + "</WirelineLog>\n");
        }
    }

    public static Curve parseCurve(Element el) {
        int curveID = Integer.parseInt(el.getChildText("CurveID"));
        String abr = el.getAttributeValue("LogDef");
        String mnemonic = el.getChildText("Mnemonic");
        String comments = el.getChildText("Comments");
        LinkedList<Curve.CurveValue> trace = new LinkedList<Curve.CurveValue>();
        for (Element row : el.getChildren("Row")) {
            double depth = Double.parseDouble(row.getAttributeValue("Depth"));
            double value = Double.parseDouble(row.getAttributeValue("Value"));
            trace.add(new Curve.CurveValue(depth, value));
        }
        return new Curve(curveID, trace, abr, mnemonic, null, comments, null);
    }

    public int getWellID() {
        return this.wellID;
    }

    private void setChanged() {
        this.hasChanged = true;
    }

    public void addListener(WellCurveService.Listener l) {
        this.listenerList.addListener((Object)l);
    }

    public void deleteListener(WellCurveService.Listener l) {
        this.listenerList.deleteListener((Object)l);
    }

    public Stream<WellCurveService.Listener> getListeners() {
        return this.listenerList.stream();
    }

    public void notifyListeners() {
        if (this.hasChanged) {
            this.listenerList.stream().forEach(WellCurveService.Listener::curveListUpdated);
            this.hasChanged = false;
        }
    }
}

