/*
    StrataBugs v3.0 Base - (H2) database creation script.

    This script will create a schema for StrataBugs 3.0. It is the source document for
    all DB creation scripts, used by SBConvert3_0.jar. 

    All foreign keys should be indexed to prevent performance issues.
*/

/* Note: to query the names of the constraints, (if they are not explicitly named) so they can be altered or dropped, use the following sql:

select c.table_name,c.constraint_name, c.r_constraint_name, cc.table_name
from all_constraints c
inner join all_constraints cc on c.r_constraint_name = cc.constraint_name
where c.table_name='<table_name>';

*/

/*
using index  CREATE INDEX ==IDX ON =table=(LOWER(==));  ???optional lowercase???
OR: CREATE INDEX ==IDX ON =table=(==); 
???Implement case-insensitive searches by putting all data into LOWER case??? e.g. SELECT * FROM my_table WHERE UPPER(empname) = 'KARL';
*/

/* Note: use BIT with 0 or 1 for boolean values, these will get converted to appropriate
    Note: Oracle queries don't support TRUE or FALSE
   column types for the platform
*/

/* This controls assignment of unique IDs */
CREATE TABLE CONTROL (
    TABLE_NAME      VARCHAR2(30) NOT NULL, 
    NEXT_ID         NUMBER(11,0),
	CONSTRAINT CONTROL_PK PRIMARY KEY (TABLE_NAME)
);

/* Users, and non-users (analysts) */
CREATE TABLE USERDEF (
    USER_ID         NUMBER(5,0),           /*Internal ID unique number*/
    NAME            VARCHAR2(40)    NOT NULL,   /* increased to 40 chars 29 Sep 2011 */
    ABR             VARCHAR2(5)     NOT NULL,
    SYS_NAME        VARCHAR2(60),   /* Increased for v3.0 */
    DISC            VARCHAR2(1),    /* M, P OR N ONLY?*/
    USER_PRIV       NUMBER(11,0),   /* zero if non-user*/
    PASSWORD        VARCHAR2(20),
    COLOUR          VARCHAR2(9),
    CREATED         DATE,           /* these fields added for v3.0 */
    CREATOR         NUMBER(5,0),    /* these are not constrained by FK to prevent self-referencing errors */
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT U_ABR UNIQUE(ABR),
    CONSTRAINT USERDEF_PK PRIMARY KEY (USER_ID)
); 

/* Wells = wellbores, separate row for each bore/sidetrack. Also contains outcrop section locations */
CREATE TABLE WELLS (
    WELL_ID         NUMBER(11,0),   /* Unique ID */    
    WELL_NAME       VARCHAR2(80),   /* Does not need to be unique */
    WELL_ALT_NAME   VARCHAR2(80),   /* Alternative well name, new for v3, not unique */
    WELL_CODE       VARCHAR2(80)    NOT NULL UNIQUE,
    DESCRIP         VARCHAR2(1000),  /* From WELL_IDENT */
    OPERATOR        VARCHAR2(50),   /* Increased to 50 13 Aug 2020 */
    CLIENT          VARCHAR2(100),  /* new for v3.0 name of client. Convenience search field for service companies */
    COUNTRY         VARCHAR2(40)    NOT NULL, /* increased 26 Sep 2019, from v2.1.1 can contain mixed case */
    LOC_STATE       VARCHAR2(40),   /* New for v3, the state, US or otherwise, of well location. */
    COUNTY          VARCHAR2(30),    
    OCS             VARCHAR2(40),   /* increased 30 Sep 2011. Also used for lease name/number */
    AREA            VARCHAR2(80),   /* area/lease block name */
    BASIN           VARCHAR2(80),   /* Sedimentary basin, new for v3 */
    FIELD           VARCHAR2(40),   /* increased 30 Sep 2011 */
    GRID_X          NUMBER(11,3),
    GRID_Y          NUMBER(11,3),   /* these are often UTMs, but not necessarily */
    UTMZONE         VARCHAR2(5),    /* these are 1-60 C-X e.g 30U. Extra char for N/S indicator */
    SPUD_DATE       DATE,
    COMP_DATE       DATE,
    LAT_DEC         NUMBER(8,6),    /* Decimal degrees, 0-90 negative for S */
    LONG_DEC        NUMBER(9,6),    /* Decimal degrees, 0-180 negative for W */
    TD              NUMBER(11,3)    NOT NULL,
    SL              NUMBER(11,3),
    SB              NUMBER(11,3),
    RTE             NUMBER(11,3),
    KICKO           NUMBER(11,3),
    DATUM           NUMBER(1,0)    NOT NULL CHECK (DATUM BETWEEN 0 AND 3),   /*0 means no info */
    TYPE            VARCHAR2(1)    CHECK (TYPE IN ('W', 'O')),      /* WORKS FOR LOWERCASE? */
    UNITS           VARCHAR2(1)    CHECK (UNITS IN ('F', 'M')),
    GEODATUM        VARCHAR2(80),  /* This should be the sphaeroid e.g. WGS84 */
    LITH_DEPTH      VARCHAR2(1),
    COLOUR          VARCHAR2(9),   /* Hex RGB colour #rrggbb(+aa) */
    DATA_SOURCE     VARCHAR2(40),  /* Source of header information */
    CHART_URL       VARCHAR2(255),
    SCALE           NUMBER(5,0),   /* Default plotting scale. Added for v3.0 */
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    ACM             NUMBER(5,0),    /* From well_ident */
    CONSTRAINT WELLS_PK PRIMARY KEY (WELL_ID),
    CONSTRAINT WELLS_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT WELLS_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT WELLS_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX WELL_NAME_IDX1 ON WELLS(WELL_NAME);
CREATE INDEX WELL_NAME_IDX2 ON WELLS(WELL_CODE); /*!NOT_ORACLE */


/* WELL_IDENT Merged with wells in v3.0 */

CREATE TABLE WELLS_ACL (     /* Formerly table well_ident_acl */
    WELL_ID         NUMBER(11,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT WELLS_ACL_PK PRIMARY KEY (WELL_ID, USER_ID),
    CONSTRAINT WELLS_ACL_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID),
    CONSTRAINT WELLS_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

/* New for v3. Relationship between well sidetracks and main tracks. Entries can be recursive */
CREATE TABLE WELL_TRACK (
    WELL_ID         NUMBER(11,0) NOT NULL,
    PARENT_ID       NUMBER(11,0) NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT WELL_TRACK_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID),
    CONSTRAINT WELL_TRACK_FK2 FOREIGN KEY (PARENT_ID) REFERENCES WELLS (WELL_ID),
);

CREATE TABLE IMAGE_SET (
    IMAGE_SET_ID    NUMBER(11,0),
    DESCR           VARCHAR2(255),
    CONSTRAINT IMAGE_SET_PK PRIMARY KEY (IMAGE_SET_ID)
); 

CREATE TABLE SBIMAGE (
    IMAGE_ID        NUMBER(11,0),
    CAPTION         VARCHAR2(255),
    PIC_PATH        VARCHAR2(255),
    IMAGE_DATA      BLOB,        /*1073741823*/    
    CONSTRAINT SBIMAGE_PK PRIMARY KEY (IMAGE_ID)
/*    CHECK (PIC_PATH NOT NULL OR IMAGE_DATA NOT NULL)*/
); 

CREATE TABLE IMAGE_SETMBR (
    IMAGE_SET_ID    NUMBER(11,0),
    IMAGE_ID        NUMBER(11,0),
    SORT            NUMBER(5,0),    /* Sort order from 1 (highest to n)*/
    CONSTRAINT IMAGE_SETMBR_PK PRIMARY KEY (IMAGE_SET_ID, IMAGE_ID),
    CONSTRAINT IMAGE_SETMBR_FK1 FOREIGN KEY (IMAGE_SET_ID) REFERENCES IMAGE_SET(IMAGE_SET_ID),
    CONSTRAINT IMAGE_SETMBR_FK2 FOREIGN KEY (IMAGE_ID) REFERENCES SBIMAGE(IMAGE_ID)
); 

CREATE TABLE SAMPLES (
    WELL_ID         NUMBER(11,0),
    SAMP_ID         NUMBER(11,0),
    TOP_DEPTH       NUMBER,
    BASE_DEPTH      NUMBER,
    TYPE            VARCHAR2(5)     NOT NULL,
    GRID_X          NUMBER,
    GRID_Y          NUMBER,
    LABEL           VARCHAR2(40),   /* increased to 40 chars 25 Jun 2025 */
    IMAGE_SET_ID    NUMBER(11,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SAMPLES_PK PRIMARY KEY (WELL_ID, SAMP_ID),
    CONSTRAINT SAMPLES_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SAMPLES_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SAMPLES_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SAMPLES_FK4 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT SAMPLES_FK5 FOREIGN KEY (IMAGE_SET_ID) REFERENCES IMAGE_SET (IMAGE_SET_ID)
); 

/* Microfossil counting Abundance schemes */
CREATE TABLE ABNSCHME (
    SCHEME_ID       NUMBER(5,0),
    NAME            VARCHAR2(25)   NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),  
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT ABNSCHME_PK  PRIMARY KEY (SCHEME_ID),
    CONSTRAINT ABNSCHME_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ABNSCHME_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ABNSCHME_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE ABNSCHMBR (
    SCHEME_ID       NUMBER(5,0),
    ENTRY_NO        NUMBER(11,0),
    ENTRY_NAME      VARCHAR2(25)   NOT NULL,
    ABR             VARCHAR2(4)    NOT NULL,
    MIN             NUMBER(11,0)   NOT NULL,
    OUTPUT_VAL      NUMBER(11,0),
    SYMBOL          VARCHAR2(20),   /* new for v3.0 */
    CONSTRAINT ABNSCHMBR_PK PRIMARY KEY (SCHEME_ID, ENTRY_NO),
    CONSTRAINT ABNSCHMBR_FK FOREIGN KEY (SCHEME_ID) REFERENCES ABNSCHME(SCHEME_ID)
); 

CREATE TABLE AUDITTRAIL (
    TRACK_DATE      TIMESTAMP,
    USER_ID         VARCHAR2(5),
    DATA_TYPE       VARCHAR2(20),
    DESCRIP         VARCHAR2(255)
);

/* Interpretation headers */
CREATE TABLE INTERP (
    INTERP_ID       NUMBER(5,0),
    DESCRIP         VARCHAR2(80)    NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT INTERP_PK PRIMARY KEY (INTERP_ID),
    CONSTRAINT INTERP_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT INTERP_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT INTERP_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
);

/* Biostratigraphic comments */
CREATE TABLE BCMMNTS (
    WELL_ID         NUMBER(11,0),       
    USAMP_ID        NUMBER(11,0),      
    LSAMP_ID        NUMBER(11,0),     
    INTERP_ID       NUMBER(5,0),      
    DISC_ID         VARCHAR2(1),
    ANALYST         NUMBER(5,0),      
    SOURCE          VARCHAR2(50),
    COMMENTS        VARCHAR2(1400),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT BCMMNTS_PK PRIMARY KEY (WELL_ID, USAMP_ID, INTERP_ID, DISC_ID, ANALYST),
    CONSTRAINT BCMMNTS_FK1 FOREIGN KEY (WELL_ID, LSAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT BCMMNTS_FK2 FOREIGN KEY (WELL_ID, USAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT BCMMNTS_FK3 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT BCMMNTS_FK4 FOREIGN KEY (ANALYST) REFERENCES USERDEF(USER_ID),
    CONSTRAINT BCMMNTS_FK5 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT BCMMNTS_FK6 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT BCMMNTS_FK7 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

/* Casing diameters reference */
CREATE TABLE CASDIAM (
    SORT_ORDER      NUMBER(5,0),
    DIAM            VARCHAR2(10) NOT NULL,
    CONSTRAINT CASDIAM_PK   PRIMARY KEY (SORT_ORDER),
    UNIQUE (DIAM)
); 

CREATE TABLE CASING (
    WELL_ID         NUMBER(11,0),
    DEPTH           NUMBER NOT NULL,
    DIAM            VARCHAR2(9) NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CASING_PK PRIMARY KEY (WELL_ID, DEPTH),
    CONSTRAINT CASING_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT CASING_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CASING_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CASING_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

/* Taxon category colours CATCOL - merged with CATEGORY in v3.0 */

/* Biostratigraphic disciplines */
CREATE TABLE DISCS (
    DISC_NAME       VARCHAR2(30)    NOT NULL,
    DISC_ABR        VARCHAR2(5)     NOT NULL,
    DISC_ADJ        VARCHAR2(30)    NOT NULL,
    DISC_ID         VARCHAR2(1),
    CONSTRAINT DISCS_PK PRIMARY KEY (DISC_ID)
); 

/* Taxon categories - each genus must belong to a category */
CREATE TABLE CATEGORY (
    DISC_ID         VARCHAR2(1),
    CAT_MNEM        VARCHAR2(5)     NOT NULL,
    CAT_NAME        VARCHAR2(30)    NOT NULL,
    COLOUR          VARCHAR2(9),   /* Hex RGB colour #rrggbb(+aa) */
    CONSTRAINT CATEGORY_PK PRIMARY KEY (CAT_MNEM),
    CONSTRAINT CATEGORY_FK FOREIGN KEY (DISC_ID) REFERENCES DISCS(DISC_ID)
); 

CREATE TABLE CHLABELS (
    GLOBAL          VARCHAR2(100),
    LOCAL           VARCHAR2(100),
    CONSTRAINT CHLABELS_PK PRIMARY KEY (GLOBAL)
); 

CREATE TABLE GENUS (
    CAT_MNEM        VARCHAR2(5)     NOT NULL,
    GEN_ID          NUMBER(11,0),
    Q1              VARCHAR2(9),
    GENUS           VARCHAR2(50)    NOT NULL,   /* increased to 50 for v3.0 */
    Q2              VARCHAR2(9),
    Q3              VARCHAR2(9),
    SUB_GENUS       VARCHAR2(30),
    Q4              VARCHAR2(9),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    COLOUR          VARCHAR2(9),    /* new for v3.0 */
    CONSTRAINT GENUS_PK PRIMARY KEY (GEN_ID),
    CONSTRAINT GENUS_FK1 FOREIGN KEY (CAT_MNEM) REFERENCES CATEGORY(CAT_MNEM),
    CONSTRAINT GENUS_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT GENUS_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT GENUS_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
);
CREATE INDEX GENUS_IDX1 ON GENUS(GENUS);
CREATE INDEX SUB_GENUS_IDX1 ON GENUS(SUB_GENUS);

CREATE TABLE SPECIES (
    GEN_ID          NUMBER(11,0)   NOT NULL,
    SPEC_ID         NUMBER(11,0),
    Q1              VARCHAR2(9),
    SPECIES         VARCHAR2(50)    NOT NULL,   /* v3.0 increased length */
    Q2              VARCHAR2(9),
    Q3              VARCHAR2(9),
    SUB_SPEC        VARCHAR2(50),    /* v3.0 increased length */
    Q4              VARCHAR2(9),
    ALPHACODE       VARCHAR2(30),
    AUTHOR          VARCHAR2(100),  /* expanded in v3.0 and merged with year field. This is a short form of the reference feld */
    REFERENCE       VARCHAR2(300),  /* in v3.0 merged from TXREFS */
    NOTES           VARCHAR2(500),  /* from TXNOTES in v3.0 */
    URL             VARCHAR2(255),  /* from TXURL in v3.0 */
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SPECIES_PK PRIMARY KEY (SPEC_ID),
    CONSTRAINT SPECIES_FK1 FOREIGN KEY (GEN_ID) REFERENCES GENUS(GEN_ID),
    CONSTRAINT SPECIES_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SPECIES_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SPECIES_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX GEN_ID_IDX1 ON SPECIES(GEN_ID);
CREATE INDEX SPECIES_IDX1 ON SPECIES(SPECIES);
CREATE INDEX SUB_SPEC_IDX1 ON SPECIES(SUB_SPEC);

/* Dictionary of events. EV_TYPE dropped for v3.0, event type is now in events and cmpstdev only */
CREATE TABLE EVENTDIC (
    EV_ID           NUMBER(11,0),
    SPEC_ID         NUMBER(11,0),
    DISC_ID         VARCHAR2(1),    /* Added for v3.0 to allow discipline selection for events that aren't linked to taxa. Should be null if spec_id is not null */
    EV_EXTENT       VARCHAR2(1),
    EV_DESC         VARCHAR2(240),
    ABR             VARCHAR2(100),
    GENERATE        BIT DEFAULT 0,      /* changed from Y/N in v2.1 */
    NAME            VARCHAR2(100) NOT NULL,     /* dropped unique key here since it's now a combination of name & prefixes */
    ABNSCH_ID       NUMBER(5,0),
    ABN_ABR         VARCHAR2(4),    /* Loose reference to abn abr entry in scheme */
    ABN             NUMBER(11,0),   /* For possible future use as abundance criterion (retained from v2.1) */
    HAS_TOP         BIT DEFAULT 1,   /* Can be used as a top event */
    HAS_BASE        BIT DEFAULT 1,   /* Can be used as a base event */
    HAS_SINGLE      BIT DEFAULT 1,   /* Can be used as a single event */
    PFX_WELL_TOP    VARCHAR2(20),    /* Prefix for when used as a top in Wells. If null, use system preferences, if empty string, use no prefix */
    PFX_WELL_BASE   VARCHAR2(20),    /* Prefix when used as base */
    PFX_WELL_SINGLE VARCHAR2(20),    /* Prefix when used as single */
    PFX_SCHEME_TOP  VARCHAR2(20),    /* Prefix for when used as a top as age i.e. in Composite Stds. */
    PFX_SCHEME_BASE VARCHAR2(20),    /* Prefix when used as base */
    PFX_SCHEME_SINGLE  VARCHAR2(20),    /* Prefix when used as single */
    PFX_SFX         BIT DEFAULT 1,  /* TRUE for use as prefix, 0 for use as suffix (Reserved for future use)*/
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT EVENTDIC_PK PRIMARY KEY (EV_ID),
    CHECK(SPEC_ID IS NOT NULL OR NAME IS NOT NULL),
    CONSTRAINT EVENTDIC_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT EVENTDIC_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT EVENTDIC_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT EVENTDIC_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT EVENTDIC_FK5 FOREIGN KEY (ABNSCH_ID) REFERENCES ABNSCHME(SCHEME_ID)
); 
CREATE INDEX EVENTDIC_IDX1 ON EVENTDIC(SPEC_ID);    /* This is necessary becuase FKs are not auto indexed */

CREATE TABLE SOURCE (
    SOURCE_ID       VARCHAR2(8),
    TXT             VARCHAR2(80)    NOT NULL,
    CONSTRAINT SOURCE_PK PRIMARY KEY (SOURCE_ID)
); 

/* new for v3.0. Provides option for translating event prefixes for output (or input) to/from another data source, so the same events can be output with alternate prefixes */
CREATE TABLE EVENT_PFX_TRANS (
    SOURCE_ID       VARCHAR2(8),
    PFX             VARCHAR2(6),
    PFX_OUT         VARCHAR2(6) NOT NULL,
    CONSTRAINT EVENT_PFX_TRANS_PK PRIMARY KEY (SOURCE_ID, PFX),
    CONSTRAINT EVENT_PFX_TRANS_FK1 FOREIGN KEY (SOURCE_ID) REFERENCES SOURCE(SOURCE_ID)
);

/* Interpreted Geological Data Schemes (Chrono-, Bio-, Seq-) */
CREATE TABLE IGD_SCH (
    SCH_ID          NUMBER(5,0),
    IGD_TYPE        NUMBER(5,0)    NOT NULL,
    DISC_ID         VARCHAR2(1),
    ARCHIVE         BIT DEFAULT 0,  /* Added for v3.0, changed name from hide to archive 26/5 */
    NAME            VARCHAR2(80)    NOT NULL,
    COMMENTS        VARCHAR2(1000), /* Added for v3.0 */
    SQTYPE          VARCHAR2(1),
    ACM             NUMBER(5,0),
    PARENT          NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_SCH_PK PRIMARY KEY (SCH_ID),
    CONSTRAINT IGD_SCH_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_SCH_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_SCH_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_SCH_FK4 FOREIGN KEY (PARENT) REFERENCES IGD_SCH (SCH_ID)
); 

CREATE TABLE IGD_SCH_ACL (
    SCH_ID          NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT IGD_SCH_ACL_PK PRIMARY KEY (SCH_ID, USER_ID),
    CONSTRAINT IGD_SCH_ACL_FK1 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID),
    CONSTRAINT IGD_SCH_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

/* Composite standards = Event schemes */
CREATE TABLE CMPSTD (
    STD_ID          NUMBER(5,0),
    NAME            VARCHAR2(100)   NOT NULL,
    COMMENTS        VARCHAR2(1000), /* Added for v3.0 */
    ARCHIVE         BIT DEFAULT 0,  /* Added for v3.0 */
    TOPCSU          NUMBER,
    BASECSU         NUMBER,
    MINAGE          NUMBER,
    MAXAGE          NUMBER,
    UNITS           VARCHAR2(20),
    ACM             NUMBER(5,0),
    PARENT          NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CMPSTD_PK PRIMARY KEY (STD_ID),
    CONSTRAINT CMPSTD_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CMPSTD_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CMPSTD_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CMPSTD_FK4 FOREIGN KEY (PARENT) REFERENCES IGD_SCH(SCH_ID)
); 

/* Events in Composite Standards */
CREATE TABLE CMPSTDEV (
    STD_ID          NUMBER(5,0),
    EV_ID           NUMBER(11,0),
    EV_TYPE         VARCHAR2(1)     CHECK (EV_TYPE IN ('F','L','S')),
    CSU             NUMBER  NOT NULL,
    CSUPLUS         NUMBER(6,3),
    CSUMINUS        NUMBER(6,3),
    COMMENTS        VARCHAR2(1000), /* Extended to 1000 for v3.0 */
    CONFIDENCE      NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CMPSTDEV_PK PRIMARY KEY (STD_ID, EV_ID, EV_TYPE),
    CONSTRAINT CMPSTDEV_FK1 FOREIGN KEY (STD_ID) REFERENCES CMPSTD(STD_ID),
    CONSTRAINT CMPSTDEV_FK2 FOREIGN KEY (EV_ID) REFERENCES EVENTDIC(EV_ID),
    CONSTRAINT CMPSTDEV_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CMPSTDEV_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CMPSTDEV_FK5 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

 /* Added for 3.0.19 to keep track of events which are NOT inherited from a parent */
CREATE TABLE CMPSTDEV_EX (
    STD_ID          NUMBER(5,0),    /* The std_id of the child cmpstd */
    EV_ID           NUMBER(11,0),
    EV_TYPE         VARCHAR2(1)     CHECK (EV_TYPE IN ('F','L','S')),
    CONSTRAINT CMPSTDEV_EX_PK PRIMARY KEY (STD_ID, EV_ID, EV_TYPE),
    CONSTRAINT CMPSTDEV_EX_FK1 FOREIGN KEY (STD_ID) REFERENCES CMPSTD(STD_ID),
    CONSTRAINT CMPSTDEV_EX_FK2 FOREIGN KEY (EV_ID) REFERENCES EVENTDIC(EV_ID)
);

CREATE TABLE CMPSTD_ACL (
    STD_ID          NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT CMPSTD_ACL_PK PRIMARY KEY (STD_ID, USER_ID),
    CONSTRAINT CMPSTD_ACL_FK1 FOREIGN KEY (STD_ID) REFERENCES CMPSTD (STD_ID),
    CONSTRAINT CMPSTD_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

/* New for v3.0. Defines the relationship between a cmpstd and its parent, from which it inherits and overrides events */
CREATE TABLE CMPSTD_PARENT (
    STD_ID          NUMBER(5,0) NOT NULL,
    PARENT_ID       NUMBER(5,0) NOT NULL,    
    CONSTRAINT CMPSTD_PARENT_PK PRIMARY KEY (STD_ID, PARENT_ID),
    CONSTRAINT CMPSTD_PARENT_FK1 FOREIGN KEY (STD_ID) REFERENCES CMPSTD(STD_ID),
    CONSTRAINT CMPSTD_PARENT_FK2 FOREIGN KEY (PARENT_ID) REFERENCES CMPSTD(STD_ID)
);

CREATE TABLE COREIMAGE (
    WELL_ID         NUMBER(11,0),
    CORENUM         VARCHAR2(3),
    TOP_DEPTH       NUMBER  NOT NULL,
    BASE_DEPTH      NUMBER  NOT NULL,
    IMAGE_ID        NUMBER(11,0),
    ISDESCRIP       NUMBER(1)   CHECK (ISDESCRIP BETWEEN 0 AND 1),
    CONSTRAINT COREIMAGE_PK PRIMARY KEY (WELL_ID, IMAGE_ID),
    CONSTRAINT COREIMAGE_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT COREIMAGE_FK2 FOREIGN KEY (IMAGE_ID) REFERENCES SBIMAGE(IMAGE_ID)
); 

CREATE TABLE CORES (
    WELL_ID         NUMBER(11,0) NOT NULL,
    TOP_DEPTH       NUMBER NOT NULL,
    BASE_DEPTH      NUMBER NOT NULL,
    CORENUM         VARCHAR2(4),
    RECOVERED       VARCHAR2(1),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CORES_PK PRIMARY KEY (WELL_ID, TOP_DEPTH),
    CONSTRAINT CORES_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT CORES_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CORES_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CORES_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE CORESHIFT (
    WELL_ID         NUMBER(11,0) NOT NULL,
    DEPTH           NUMBER NOT NULL,
    SHIFT           NUMBER NOT NULL,
    CONSTRAINT CORESHIFT_PK PRIMARY KEY (WELL_ID, DEPTH),
    CONSTRAINT CORESHIFT_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
);   

/* Hydrocarbon shows. New for v3.0 */
CREATE TABLE HC_SHOWS ( 
    WELL_ID         NUMBER(11,0) NOT NULL,
    DEPTH           NUMBER NOT NULL,
    TYPE            VARCHAR2(20),
    COMMENTS        VARCHAR2(255),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT HC_SHOWS_PK PRIMARY KEY (WELL_ID, DEPTH),
    CONSTRAINT HC_SHOWS_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT HC_SHOWS_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT HC_SHOWS_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT HC_SHOWS_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE ENV_SCH (
    ENVSCH_ID       NUMBER(5,0),
    NAME            VARCHAR2(40)    NOT NULL,
    ACM             NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT ENV_SCH_PK PRIMARY KEY (ENVSCH_ID),
    CONSTRAINT ENV_SCH_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ENV_SCH_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ENV_SCH_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE ENV_SCH_ACL (
    ENVSCH_ID       NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT ENV_SCH_ACL_PK PRIMARY KEY (ENVSCH_ID, USER_ID),
    CONSTRAINT ENV_SCH_ACL_FK1 FOREIGN KEY (ENVSCH_ID) REFERENCES ENV_SCH (ENVSCH_ID),
    CONSTRAINT ENV_SCH_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE ENVSCHMBR (
    ENVSCH_ID       NUMBER(5,0),
    TERM_NO         NUMBER(3,0),
    TERM            VARCHAR2(80) NOT NULL,
    COLOUR          VARCHAR2(9),
    COMMENTS        VARCHAR2(255),
    CONSTRAINT ENVSCHMBR_PK PRIMARY KEY (ENVSCH_ID, TERM_NO),
    CONSTRAINT ENVSCHMBR_FK1 FOREIGN KEY (ENVSCH_ID) REFERENCES ENV_SCH(ENVSCH_ID)
); 

CREATE TABLE ENVTRN (
    SOURCE_ID       VARCHAR2(8),
    ENVSCH_ID       NUMBER(5,0),
    TXT             VARCHAR2(150)   NOT NULL,
    PROX_ID         NUMBER(5,0)    NOT NULL,
    DIST_ID         NUMBER(5,0)    NOT NULL,
    CONSTRAINT ENVTRN_PK PRIMARY KEY (SOURCE_ID, ENVSCH_ID),
    CONSTRAINT ENVTRN_FK1 FOREIGN KEY (SOURCE_ID) REFERENCES SOURCE(SOURCE_ID),
    CONSTRAINT ENVTRN_FK2 FOREIGN KEY (ENVSCH_ID) REFERENCES ENV_SCH(ENVSCH_ID)
); 
CREATE INDEX PROX_ID_IDX1 ON ENVTRN(PROX_ID);
CREATE INDEX DIST_ID_IDX1 ON ENVTRN(DIST_ID);


CREATE TABLE ANALY_HDR (
    ANALY_ID        NUMBER(11,0),       
    WELL_ID         NUMBER(11,0),
    DISC_ID         VARCHAR2(1)     NOT NULL,
    ANALYST         NUMBER(5,0)    NOT NULL,      /* Analyst Link to usrid field in USERDEF*/
    ANALY_NO        NUMBER(11,0)   NOT NULL,           /* Sequential number for multiple analyses by the same analyst at different periods */
    DATE_FROM       DATE,
    DATE_TO         DATE,
    DEPTH_FROM      NUMBER,    
    DEPTH_TO        NUMBER,    
    COMMENTS        VARCHAR2(255),    
    ENVSCH_ID       NUMBER(5,0),
    ABNSCH_ID       NUMBER(5,0),
    COLOUR          VARCHAR2(9),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),      
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),      
    CONSTRAINT ANALY_HDR_PK PRIMARY KEY (ANALY_ID, WELL_ID),
    CONSTRAINT ANALY_HDR_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT ANALY_HDR_FK2 FOREIGN KEY (ENVSCH_ID) REFERENCES ENV_SCH(ENVSCH_ID),
    CONSTRAINT ANALY_HDR_FK3 FOREIGN KEY (ANALYST) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ANALY_HDR_FK4 FOREIGN KEY (ABNSCH_ID) REFERENCES ABNSCHME(SCHEME_ID),
    CONSTRAINT ANALY_HDR_FK5 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ANALY_HDR_FK6 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT ANALY_HDR_FK7 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    UNIQUE (WELL_ID, DISC_ID, ANALYST, ANALY_NO)
);  
CREATE INDEX DISC_ID_IDX1 ON ANALY_HDR(DISC_ID); 

/* Events occurring in wells/outcrops */
CREATE TABLE EVENTS (
    INTERP_ID       NUMBER(5,0),
    EV_ID           NUMBER(11,0),
    EV_TYPE         VARCHAR2(1)         CHECK (EV_TYPE IN ('F','L','S')),
    WELL_ID         NUMBER(11,0),
    SAMP_ID         NUMBER(11,0),
    DISC_ID         VARCHAR2(1),
    ANALYST         NUMBER(5,0),
    QUALIFIER       VARCHAR2(16),
    COMMENTS        VARCHAR2(240),
    SOURCE          VARCHAR2(5),
    COMPANY         VARCHAR2(20),
    CONFIDENCE      NUMBER(5,0) DEFAULT 0 NOT NULL,
    DEPTHPLUS       NUMBER(7,3),
    DEPTHMINUS      NUMBER(7,3),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT EVENTS_PK PRIMARY KEY (INTERP_ID, EV_ID, EV_TYPE, WELL_ID, SAMP_ID),
    CONSTRAINT EVENTS_FK1 FOREIGN KEY (WELL_ID, SAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),    /*CHECK THIS IS A GOOD IDEA*/
    CONSTRAINT EVENTS_FK2 FOREIGN KEY (EV_ID) REFERENCES EVENTDIC(EV_ID),
    CONSTRAINT EVENTS_FK3 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT EVENTS_FK4 FOREIGN KEY (ANALYST) REFERENCES USERDEF(USER_ID),
    CONSTRAINT EVENTS_FK5 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT EVENTS_FK6 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT EVENTS_FK7 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
);
CREATE INDEX DISC_ID_IDX3 ON EVENTS(DISC_ID);

/* Lookup table for loading previously matched events */
CREATE TABLE EVLOAD (
    SOURCE_ID       VARCHAR2(8),
    EV_ID           NUMBER(11,0),
    TXT             VARCHAR2(150),
    CONSTRAINT EVLOAD_PK PRIMARY KEY (SOURCE_ID, TXT),
    CONSTRAINT EVLOAD_FK1 FOREIGN KEY (EV_ID) REFERENCES EVENTDIC (EV_ID),
    CONSTRAINT EVLOAD_FK2 FOREIGN KEY (SOURCE_ID) REFERENCES SOURCE (SOURCE_ID)
);

CREATE TABLE FAULTS (
    WELL_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    SAMP_ID         NUMBER(11,0),
    THROW           NUMBER(8,3),
    LABEL           VARCHAR2(80),
    BND             NUMBER(1) NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT FAULTS_PK PRIMARY KEY (WELL_ID, INTERP_ID, SAMP_ID),
    CONSTRAINT FAULTS_FK1 FOREIGN KEY (WELL_ID, SAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT FAULTS_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT FAULTS_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT FAULTS_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT FAULTS_FK5 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
);

/* new for v3 to allow projects to include well lists */
CREATE TABLE PROJECT (
    PROJ_ID         NUMBER(5,0),
    NAME            VARCHAR2(60)    NOT NULL,
    DESCR           VARCHAR2(500),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT PROJECT_PK PRIMARY KEY (PROJ_ID),
    CONSTRAINT PROJECT_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT PROJECT_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT PROJECT_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
);

/* Renamed from SBWLLST in v3.0 */
CREATE TABLE WELLIST (
    WELLIST_ID      NUMBER(5,0),
    PROJ_ID         NUMBER(5,0)     NOT NULL,
    NAME            VARCHAR2(60)    NOT NULL,
    DESCR           VARCHAR2(500),  /* Added this for v3.0 */
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    UNIQUE (WELLIST_ID, PROJ_ID), /* needed for postgresql to use as a foreign key */
    CONSTRAINT WELLIST_PK PRIMARY KEY (WELLIST_ID),
    CONSTRAINT WELLIST_FK1 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID),
    CONSTRAINT WELLIST_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT WELLIST_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT WELLIST_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE WELLIST_MBR (
    WELLIST_ID      NUMBER(5,0),
    WELL_ID         NUMBER(11,0),
    CONSTRAINT WELLIST_MBR_PK PRIMARY KEY (WELLIST_ID, WELL_ID),
    CONSTRAINT WELLIST_MBR_FK1 FOREIGN KEY (WELLIST_ID) REFERENCES WELLIST(WELLIST_ID),
    CONSTRAINT WELLIST_MBR_FK2 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
); 

CREATE TABLE TXGROUP (
    GRP_ID          NUMBER(5,0),
    NAME            VARCHAR2(30)    NOT NULL,
    ABR             VARCHAR2(10),
    PROJ_ID         NUMBER(5,0),
    DESCR           VARCHAR(255),
    COLOUR          VARCHAR2(9),   /* Hex RGB colour #rrggbb(+aa) */
    ACM             NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT TXGROUP_PK PRIMARY KEY (GRP_ID),
    CONSTRAINT TXGROUP_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT TXGROUP_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT TXGROUP_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT TXGROUP_FK4 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID)
); 

CREATE TABLE TXGROUP_ACL (
    GRP_ID          NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT TXGROUP_ACL_PK_PK PRIMARY KEY (GRP_ID, USER_ID),
    CONSTRAINT TXGROUP_ACL_FK1 FOREIGN KEY (GRP_ID) REFERENCES TXGROUP (GRP_ID),
    CONSTRAINT TXGROUP_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE GROUPMBR (
    SPEC_ID         NUMBER(11,0),
    GRP_ID          NUMBER(5,0),
    CONSTRAINT GROUPMBR_PK PRIMARY KEY (SPEC_ID, GRP_ID),
    CONSTRAINT GROUPMBR_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT GROUPMBR_FK2 FOREIGN KEY (GRP_ID) REFERENCES TXGROUP(GRP_ID)
); 

CREATE TABLE GROUPMBR_GENUS (
    GEN_ID          NUMBER(11,0)   NOT NULL,
    GRP_ID          NUMBER(5,0)    NOT NULL,
    CONSTRAINT GROUPMBR_GENUS_PK PRIMARY KEY (GEN_ID, GRP_ID),
    CONSTRAINT GROUPMBR_GENUS_FK1 FOREIGN KEY (GEN_ID) REFERENCES GENUS (GEN_ID),
    CONSTRAINT GROUPMBR_GENUS_FK2 FOREIGN KEY (GRP_ID) REFERENCES TXGROUP (GRP_ID)
);

CREATE TABLE GROUPSET (
    GRPSET_ID       NUMBER(5,0),
    NAME            VARCHAR2(30)    NOT NULL,
    PROJ_ID         NUMBER(5,0),
    DESCR           VARCHAR2(255),
    ACM             NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT GROUPSET_PK PRIMARY KEY (GRPSET_ID),
    CONSTRAINT GROUPSET_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT GROUPSET_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT GROUPSET_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT GROUPSET_FK4 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID)
); 

CREATE TABLE GROUPSET_ACL (
    GRPSET_ID       NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT GROUPSET_ACL_PK PRIMARY KEY (GRPSET_ID, USER_ID),
    CONSTRAINT GROUPSET_ACL_FK1 FOREIGN KEY (GRPSET_ID) REFERENCES GROUPSET (GRPSET_ID),
    CONSTRAINT GROUPSET_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE IGD_DICT (
    IGD_ID          NUMBER(11,0),
    SCH_ID          NUMBER(5,0)    NOT NULL,
    IGD_TYPE        NUMBER(5,0)    NOT NULL,
    HIER            NUMBER(3,0)    NOT NULL,   /* changed from 5 to 3 29 Oct 2014 */
    NAME            VARCHAR2(80)   NOT NULL,   
    ABR             VARCHAR2(80),
    U_AGE           NUMBER,
    L_AGE           NUMBER,
    COLOUR          VARCHAR2(9),
    U_UNIT_REF      NUMBER(11,0),
    U_UNIT_VAL      NUMBER(3,0)         CHECK (U_UNIT_VAL BETWEEN 0 AND 100),
    U_CONF          NUMBER(3,0)         CHECK (U_CONF BETWEEN 0 AND 100),       /* Note: the order of U_CONF got changed 13/8/25 resulting in error for loading the u_conf table. Fixed 31 Oct 25 */
    L_UNIT_REF      NUMBER(11,0),
    L_UNIT_VAL      NUMBER(3,0)         CHECK (L_UNIT_VAL BETWEEN 0 AND 100),
    L_CONF          NUMBER(3,0)         CHECK (L_CONF BETWEEN 0 AND 100),
    COMMENTS        VARCHAR2(255),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_DICT_PK PRIMARY KEY (IGD_ID),
    CONSTRAINT IGD_DICT_FK1 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT IGD_DICT_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_DICT_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_DICT_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_DICT_FK5 FOREIGN KEY (U_UNIT_REF) REFERENCES IGD_DICT(IGD_ID),
    CONSTRAINT IGD_DICT_FK6 FOREIGN KEY (L_UNIT_REF) REFERENCES IGD_DICT(IGD_ID),
    CONSTRAINT IGD_DICT_UQNAME UNIQUE (SCH_ID, HIER, NAME)  /* New for v3.0 */
); 
CREATE INDEX SCH_ID_IDX ON IGD_DICT(SCH_ID);
CREATE INDEX IGD_TYPE_IDX1 ON IGD_DICT(IGD_TYPE);
CREATE INDEX IGD_DICT_HIER ON IGD_DICT(HIER);


CREATE TABLE IGD (
    WELL_ID         NUMBER(11,0),
    IGD_TYPE        NUMBER(5,0),
    HIER            NUMBER(5,0)    NOT NULL,
    TOP_ID          NUMBER(11,0),
    BASE_ID         NUMBER(11,0),
    TOP_BND         NUMBER(3,0),
    BASE_BND        NUMBER(3,0),
    SCH_ID          NUMBER(5,0),
    Q_UZONE         VARCHAR2(1),
    UPP_ZONE        NUMBER(11,0),
    UPP_INF         VARCHAR2(80),
    Q_LZONE         VARCHAR2(1),
    LOW_ZONE        NUMBER(11,0),
    LOW_INF         VARCHAR2(80),
    INTERP_ID       NUMBER(5,0),
    TOP_ANALY       NUMBER(5,0),
    BASE_ANALY      NUMBER(5,0),
    SEP             VARCHAR2(10),
    CONFIDENCE      NUMBER(5,0) DEFAULT 0 NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_PK PRIMARY KEY (WELL_ID, IGD_TYPE, HIER, TOP_ID, SCH_ID, INTERP_ID),
    CONSTRAINT IGD_FK1 FOREIGN KEY (WELL_ID, BASE_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT IGD_FK2 FOREIGN KEY (WELL_ID, TOP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT IGD_FK3 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT IGD_FK4 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT IGD_FK5 FOREIGN KEY (UPP_ZONE) REFERENCES IGD_DICT(IGD_ID),
    CONSTRAINT IGD_FK6 FOREIGN KEY (LOW_ZONE) REFERENCES IGD_DICT(IGD_ID),
    CONSTRAINT IGD_FK7 FOREIGN KEY (TOP_ANALY) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_FK8 FOREIGN KEY (BASE_ANALY) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_FK9 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_FK10 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_FK11 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX BASE_ID_IDX1 ON IGD(BASE_ID);
CREATE INDEX UPP_ZONE_IDX ON IGD(UPP_ZONE);
CREATE INDEX LOW_ZONE_IDX ON IGD(LOW_ZONE);


CREATE TABLE IGD_DEF (
    NAME            VARCHAR2(30)    NOT NULL,
    IGD_TYPE        NUMBER(5,0),
    PARENT          NUMBER(5,0),
    HIER            NUMBER(5,0),
    CONSTRAINT IGD_DEF_PK PRIMARY KEY (IGD_TYPE, HIER)
); 

CREATE TABLE IGD_ENV (
    WELL_ID         NUMBER(11,0),
    TOP_ID          NUMBER(11,0),
    BASE_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    ENVSCH_ID       NUMBER(5,0)    NOT NULL,
    UP              NUMBER(3,0)    NOT NULL,
    UD              NUMBER(3,0)    NOT NULL,
    LP              NUMBER(3,0)    NOT NULL,
    LD              NUMBER(3,0)    NOT NULL,
    QUAL            NUMBER(3,0),
    CONFIDENCE      NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_ENV_PK PRIMARY KEY (WELL_ID, TOP_ID, INTERP_ID),
    CONSTRAINT IGD_ENV_FK1 FOREIGN KEY (WELL_ID, BASE_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT IGD_ENV_FK2 FOREIGN KEY (WELL_ID, TOP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT IGD_ENV_FK3 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT IGD_ENV_FK4 FOREIGN KEY (ENVSCH_ID) REFERENCES ENV_SCH(ENVSCH_ID),
    CONSTRAINT IGD_ENV_FK5 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_ENV_FK6 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_ENV_FK7 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX BASE_ID_IDX2 ON IGD_ENV(BASE_ID);
CREATE INDEX ENVSCH_ID_IDX1 ON IGD_ENV(ENVSCH_ID);

CREATE TABLE IGD_HDR (
    WELL_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    SCH_ID          NUMBER(5,0),
    SOURCE          VARCHAR2(25),
    AUTHORISED      VARCHAR2(25),
    COMMENTS        VARCHAR2(240),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_HDR_PK PRIMARY KEY (WELL_ID, INTERP_ID, SCH_ID),
    CONSTRAINT IGD_HDR_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT IGD_HDR_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT IGD_HDR_FK3 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT IGD_HDR_FK4 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_HDR_FK5 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_HDR_FK6 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

/* IGD_EV: This table stores event definitions for and within units (biozones) */
/* Now dropped from v3.0 as it was not used, and is likely to be linked to composites */

CREATE TABLE LITHDESC (
    LITH_ID         NUMBER(5,0),
    DESCR           VARCHAR2(80)    NOT NULL,
    ABR             VARCHAR2(10),
    CMP             VARCHAR2(1),
    GRAIN_SIZE      NUMBER,
    SORT_ORDER      NUMBER(5,0),
    CONSTRAINT LITHDESC_PK PRIMARY KEY (LITH_ID)
); 

CREATE TABLE IGD_DICT_LSTRAT (
    UNIT_ID         NUMBER(11,0) PRIMARY KEY NOT NULL,
    SCH_ID          NUMBER(5,0) NOT NULL,
    HIER            NUMBER(3,0) NOT NULL, 
    NAME            VARCHAR2(80) NOT NULL, /* Need to set this to not allow zero length! */
    ABR             VARCHAR2(80),
    UW_AGE          NUMBER,           /* This changed from NUMBER(11,3) to NUMBER 13 Jul 2023 */
    UE_AGE          NUMBER,
    LW_AGE          NUMBER,
    LE_AGE          NUMBER,
    UW_X            NUMBER(11,3),
    UE_X            NUMBER(11,3),
    LW_X            NUMBER(11,3),
    LE_X            NUMBER(11,3),
    E_BND           NUMBER(3,0) NOT NULL,    /* new for v3.0 for Shazam lines 0 - 100 min/max */
    W_BND           NUMBER(3,0) NOT NULL,
    COLOUR          VARCHAR2(9),
    LITH_ID         NUMBER(5,0),
    U_BND_INT       NUMBER(5,0) NOT NULL,
    L_BND_INT       NUMBER(5,0) NOT NULL,
    CREATED         DATE, 
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_DICT_LSTRAT_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_DICT_LSTRAT_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_DICT_LSTRAT_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_DICT_LSTRAT_FK4 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT IGD_DICT_LSTRAT_FK5 FOREIGN KEY (LITH_ID) REFERENCES LITHDESC(LITH_ID)
);

CREATE TABLE IGD_LSTRAT (
    WELL_ID         NUMBER(11,0)   NOT NULL,
    IGD_TYPE        NUMBER(5,0), /*Leave this in for future-proofing*/
    HIER            NUMBER(5,0)    CHECK (HIER >= 1 and HIER <=8),  /* Added check constraint 12 Feb 2025 */
    TOP_ID          NUMBER(11,0)   NOT NULL,
    BASE_ID         NUMBER(11,0)   NOT NULL,
    TOP_BND         NUMBER(3,0)    NOT NULL,
    BASE_BND        NUMBER(3,0)   NOT NULL,
    SCH_ID          NUMBER(5,0)    NOT NULL,
    Q_UZONE         VARCHAR2(1),
    UPP_ZONE        NUMBER(11,0),
    UPP_INF         VARCHAR2(80),
    Q_LZONE         VARCHAR2(1),
    LOW_ZONE        NUMBER(11,0),
    LOW_INF         VARCHAR2(80),
    INTERP_ID       NUMBER(5,0)    NOT NULL,
    SEP             VARCHAR2(10),
    CONFIDENCE      NUMBER(5,0) DEFAULT 0 NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT IGD_LSTRAT_PK PRIMARY KEY (WELL_ID, IGD_TYPE, HIER, TOP_ID, SCH_ID, INTERP_ID),
    CONSTRAINT IGD_LSTRAT_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_LSTRAT_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_LSTRAT_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT IGD_LSTRAT_FK4 FOREIGN KEY (WELL_ID, TOP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT IGD_LSTRAT_FK5 FOREIGN KEY (WELL_ID, BASE_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT IGD_LSTRAT_FK6 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT IGD_LSTRAT_FK7 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT IGD_LSTRAT_FK8 FOREIGN KEY (UPP_ZONE) REFERENCES IGD_DICT_LSTRAT(UNIT_ID),
    CONSTRAINT IGD_LSTRAT_FK9 FOREIGN KEY (LOW_ZONE) REFERENCES IGD_DICT_LSTRAT(UNIT_ID)
); 

/* new for v3.0 table to store Source/Reservoir/Seal units */
CREATE TABLE SRS_UNITS (
    UNIT_ID         NUMBER(11,0) PRIMARY KEY NOT NULL,
    SCH_ID          NUMBER(5,0) NOT NULL,   /* Must be a lithostrat scheme */
    TYPE            VARCHAR2(20) NOT NULL,  /* Source/Reservioir/Seal */
    NAME            VARCHAR2(80) NOT NULL,
    ABR             VARCHAR2(20),
    COLOUR          VARCHAR2(9),
    LSTRAT_UNIT_ID  NUMBER(11,0) NULL,  /* Linked lithostrat unit, can be null, but should not be */
    U_AGE           NUMBER,             /* This changed from NUMBER(11,3) to NUMBER 13 Jul 2023 */  
    L_AGE           NUMBER,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SRS_UNITS_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SRS_UNITS_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SRS_UNITS_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SRS_UNITS_FK4 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT SRS_UNITS_FK5 FOREIGN KEY (LSTRAT_UNIT_ID) REFERENCES IGD_DICT_LSTRAT(UNIT_ID)
);

/* new for v3.0 table to store Source/Reservoir/Seal zones in wells by depth rather than sample */
CREATE TABLE SRS_ZONES (
    WELL_ID         NUMBER(11,0)   NOT NULL,
    SCH_ID          NUMBER(5,0)    NOT NULL,
    INTERP_ID       NUMBER(5,0)    NOT NULL,
    UNIT_ID         NUMBER(11,0)   NOT NULL,
    INF_NAME        VARCHAR2(80),   
    TOP_DEPTH       NUMBER         NOT NULL,
    BASE_DEPTH      NUMBER         NOT NULL,
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SRS_ZONES_PK PRIMARY KEY (WELL_ID, INTERP_ID, UNIT_ID),
    CONSTRAINT SRS_ZONES_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SRS_ZONES_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SRS_ZONES_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SRS_ZONES_FK4 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT SRS_ZONES_FK5 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT SRS_ZONES_FK7 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID)
);

CREATE TABLE AGE_CURVE (
    CURVE_ID        NUMBER(5,0),
    SCH_ID          NUMBER(5,0)     NOT NULL,
    TITLE           VARCHAR2(80),
    UNITS           VARCHAR2(25),
    LINEWIDTH       NUMBER(5,3),
    COLOUR          VARCHAR2(9),
    STYLE           NUMBER(3,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT AGE_CURVE_PK PRIMARY KEY (CURVE_ID),
    CONSTRAINT AGE_CURVE_FK1 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT AGE_CURVE_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT AGE_CURVE_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT AGE_CURVE_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID)
);

CREATE TABLE AGE_TRACE (
    CURVE_ID        NUMBER(5,0),
    AGE             NUMBER NOT NULL,
    TRACE_VALUE     NUMBER,
    CONSTRAINT AGE_TRACE_PK PRIMARY KEY (CURVE_ID, AGE, TRACE_VALUE),
    CONSTRAINT AGE_TRACE_FK1 FOREIGN KEY (CURVE_ID) REFERENCES AGE_CURVE(CURVE_ID)

);

CREATE TABLE IGD_COLMAP (
    WELL_ID         NUMBER(11,0) NOT NULL,
    INTERP_ID       NUMBER(5,0) NOT NULL,
    IGD_TYPE        NUMBER(3,0) NOT NULL,
    COL             NUMBER(3,0),
    TITLE           VARCHAR2(80),
    MAXHIER         NUMBER(3,0),
    CONSTRAINT IGD_COLMAP_PK PRIMARY KEY (WELL_ID, INTERP_ID, IGD_TYPE, COL), /* col added 15 Feb 2012 */
    CONSTRAINT IGD_COLMAP_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT IGD_COLMAP_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID)
);
 
CREATE TABLE INTCMMNTS (
    WELL_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    IGD_TYPE        NUMBER(5,0) NOT NULL,
    TOP_DEPTH       NUMBER NOT NULL,
    BASE_DEPTH      NUMBER NOT NULL,
    COMMENTS        VARCHAR2(2000)  NOT NULL,   /* Changed to 2000 9 Mar 2023 */
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT INTCMMNTS_PK PRIMARY KEY (WELL_ID, INTERP_ID, TOP_DEPTH, IGD_TYPE),
    CONSTRAINT INTCMMNTS_FK1 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT INTCMMNTS_FK2 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT INTCMMNTS_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT INTCMMNTS_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT INTCMMNTS_FK5 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
);

/* Describes the key used in the next table */
CREATE TABLE INTERVAL_LOG_DEF (
    ABR             VARCHAR2(10) NOT NULL,
    DESCR           VARCHAR2(255),
    CONSTRAINT INTERVAL_LOG_DEF_PK PRIMARY KEY (ABR)
);

/* This table is designed for Chromalog data but can be used for other purposes */
CREATE TABLE INTERVAL_LOG_TRACE (
    WELL_ID         NUMBER(11,0)    NOT NULL,
    ABR             VARCHAR2(10)    NOT NULL,
    TOP_DEPTH       NUMBER(11,3)    NOT NULL,
    BASE_DEPTH      NUMBER(11,3)    NOT NULL,
    COLOUR          VARCHAR2(9),
    TXT             VARCHAR2(255),
    CONSTRAINT INTERVAL_LOG_TRACE_PK PRIMARY KEY (WELL_ID, ABR, TOP_DEPTH),
    CONSTRAINT INTERVAL_LOG_TRACE_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID),
    CONSTRAINT INTERVAL_LOG_TRACE_FK2 FOREIGN KEY (ABR) REFERENCES INTERVAL_LOG_DEF (ABR),
);

/* Storage table for IPS derived paleobathymetric curve */
CREATE TABLE IPS_BATHY (
    WELL_ID         NUMBER(11,0) NOT NULL,
    DEPTH           NUMBER(11,3) NOT NULL,
    DIVERS          NUMBER(5,3),
    BATH            NUMBER(5,3),
    BAVG            NUMBER(5,3),
    BLW             NUMBER(3,0),
    BNO             NUMBER(3,0),
    OTS             NUMBER(3,0),
    COS             NUMBER(3,0),
    TOTAL           NUMBER(5,0),
    CONSTRAINT IPS_BATHY_PK PRIMARY KEY (WELL_ID,DEPTH),
    CONSTRAINT IPS_BATHY_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
);

/* the LICENCE table has been dropped for v3.0 as it's now redundant and we don't store licensing information in the database */

CREATE TABLE LOC (
    WELL_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    STD_ID          NUMBER(5,0),
    COLOUR          VARCHAR2(9),
    LINEWIDTH       NUMBER,
    COMMENTS        VARCHAR2(255),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT LOC_PK PRIMARY KEY (WELL_ID, INTERP_ID),
    CONSTRAINT LOC_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT LOC_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT LOC_FK3 FOREIGN KEY (STD_ID) REFERENCES CMPSTD(STD_ID),
    CONSTRAINT LOC_FK4 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT LOC_FK5 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT LOC_FK6 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE LOCNODE (
    WELL_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    NODE_KEY        VARCHAR2(15),
    DEPTH           NUMBER       NOT NULL,
    AGE             NUMBER       NOT NULL,      /* This changed from NUMBER(11,3) to NUMBER 13 Jul 2023 */
    BND             NUMBER(2,0)  NOT NULL,
    COLOUR          VARCHAR2(9),
    CONSTRAINT LOCNODE_PK PRIMARY KEY (WELL_ID, INTERP_ID, NODE_KEY),
    CONSTRAINT LOCNODE_FK1 FOREIGN KEY (WELL_ID, INTERP_ID) REFERENCES LOC(WELL_ID, INTERP_ID)
); 

CREATE TABLE LOGDEF (
    ABR             VARCHAR2(10)    NOT NULL,
    TITLE           VARCHAR2(30)    NOT NULL,
    UNITS           VARCHAR2(10)    NOT NULL,
    MINVAL          NUMBER      NOT NULL,
    MAXVAL          NUMBER      NOT NULL,
    ASLOG           VARCHAR2(1)     NOT NULL,
    BACKCOLOUR      NUMBER, /* Note: non-standard colours which may be system dependent */
    LINECOLOUR      NUMBER,
    PENSTYLE        NUMBER,
    TRACESTYLE      NUMBER,
    PENWIDTH        NUMBER,
    WRAP            VARCHAR2(1),
    CONSTRAINT LOGDEF_PK PRIMARY KEY (ABR)
); 

CREATE TABLE LOG_CURVE (
    CURVE_ID        NUMBER(11,0),
    WELL_ID         NUMBER(11,0)   NOT NULL,
    ABR             VARCHAR2(10),     /*Links to logdef.abr*/
    CURVE_MNEM      VARCHAR2(50),
    FILENAME        VARCHAR2(255),
    COMMENTS        VARCHAR2(255),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT LOG_CURVE_PK PRIMARY KEY (CURVE_ID),
    CONSTRAINT LOG_CURVE_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT LOG_CURVE_FK2 FOREIGN KEY (ABR) REFERENCES LOGDEF(ABR),
    CONSTRAINT LOG_CURVE_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT LOG_CURVE_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT LOG_CURVE_FK5 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX WELL_ID_IDX1 ON LOG_CURVE(WELL_ID);


CREATE TABLE LOG_TRACE (
    CURVE_ID        NUMBER(11,0),
    DEPTH           NUMBER NOT NULL,
    TRACE_VALUE     NUMBER,
    CONSTRAINT LOG_TRACE_PK PRIMARY KEY (CURVE_ID, DEPTH),
    CONSTRAINT LOG_TRACE_FK1 FOREIGN KEY (CURVE_ID) REFERENCES LOG_CURVE(CURVE_ID)
); 

CREATE TABLE LTHTRNSCH (
    SCH_ID          NUMBER(5,0),
    DESCR           VARCHAR2(80)    NOT NULL,
    CREATOR         VARCHAR2(5),
    CREATED         DATE,
    USECODES        VARCHAR2(1)     NOT NULL,
    CONSTRAINT LTHTRNSCH_PK PRIMARY KEY (SCH_ID)
); 

CREATE TABLE OVERLAY (
    OVR_ID          NUMBER(5,0),
    NAME            VARCHAR2(25)    NOT NULL,
    DESCRIP         VARCHAR2(240),
    MAXTEXT         VARCHAR2(1),
    OVROWS          NUMBER(5,0),
    OVCOLUMNS       NUMBER(5,0),
    USECODES        VARCHAR2(1),    /* This attribute added 29 Apr 2013 */
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT OVERLAY_PK PRIMARY KEY (OVR_ID),
    CONSTRAINT OVERLAY_FK1 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT OVERLAY_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT OVERLAY_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE OVR_MAP (
    OVR_ID          NUMBER(5,0),
    OVROW           NUMBER(5,0)    NOT NULL,
    OVCOL           NUMBER(5,0)    NOT NULL,
    SPEC_ID         NUMBER(11,0)   NOT NULL,
    CONSTRAINT OVR_MAP_PK PRIMARY KEY (OVR_ID, OVROW, OVCOL),
    CONSTRAINT OVR_MAP_FK1 FOREIGN KEY (OVR_ID) REFERENCES OVERLAY(OVR_ID),
    CONSTRAINT OVR_MAP_FK2 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID)
); 
CREATE INDEX SPEC_ID_IDX2 ON OVR_MAP(SPEC_ID);


CREATE TABLE PREF_SYSTEM (
    PREF_KEY        VARCHAR2(50)    NOT NULL,
    PREF_VALUE      VARCHAR2(255)   NOT NULL,
    CONSTRAINT PREF_SYSTEM_PK PRIMARY KEY (PREF_KEY, PREF_VALUE)
); 

CREATE TABLE PREF_USER (
    USER_ID         NUMBER(5,0),
    PREF_KEY        VARCHAR2(12)    NOT NULL,	/* increased to 12 for v3 */
    PREF_VALUE      VARCHAR2(255)   NOT NULL,
    CONSTRAINT PREF_USER_PK PRIMARY KEY (USER_ID, PREF_KEY),
    CONSTRAINT PREF_USER_FK1 FOREIGN KEY (USER_ID) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE SBCHARTS ( /* Obsolete in 3.0 but retained for legacy chart data */
    CHART_ID        NUMBER(11,0),
    GLOBAL          VARCHAR2(1),
    OWNER           VARCHAR2(5),
    WELL_ID         NUMBER(11,0),
    PROJECT         VARCHAR2(50),
    NAME            VARCHAR2(100),  /* Increased 10 Jan 2024 to accommodate some old padded data */
    TOP_DEPTH       NUMBER,
    BASE_DEPTH      NUMBER,
    SCALE           NUMBER,
    UNITS           VARCHAR2(1),
    HDR_HEIGHT      NUMBER,
    MONO_LITH       NUMBER,
    INTERP_ID       NUMBER(5,0),
    SYN_SCH         NUMBER(5,0),
    ASYN_SCH        NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SBCHARTS_PK PRIMARY KEY (CHART_ID),
    CONSTRAINT SBCHARTS_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT SBCHARTS_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SBCHARTS_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SBCHARTS_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX WELL_ID_IDX2 ON SBCHARTS(WELL_ID);

/* Obsolete for v3 except for legacy charts */
CREATE TABLE SBCHCLIN (
    CHART_ID        NUMBER(11,0),
    LINE_NO         NUMBER(5,0),
    RED             NUMBER(3,0)      CHECK (RED BETWEEN 0 AND 255),
    GREEN           NUMBER(3,0)      CHECK (GREEN BETWEEN 0 AND 255),
    BLUE            NUMBER(3,0)      CHECK (BLUE BETWEEN 0 AND 255),
    DATUM           VARCHAR2(1),
    STYLE           NUMBER(5,0),
    ID              NUMBER(5,0),
    INTERP_ID       NUMBER(5,0),
    SURFACE         VARCHAR2(1),
    WIDTH           NUMBER,
    FONT_SIZE       NUMBER,
    LABEL           VARCHAR2(1),
    PLOTENDS        NUMBER(3,0),
    CONSTRAINT SBCHCLIN_PK PRIMARY KEY (CHART_ID, LINE_NO),
    CONSTRAINT SBCHCLIN_FK1 FOREIGN KEY (CHART_ID) REFERENCES SBCHARTS(CHART_ID)
); 

/* Obsolete for v3 except for legacy charts */
CREATE TABLE SBCHPANL (
    CHART_ID        NUMBER(11,0),
    PANEL_NO        NUMBER(5,0),
    ATTRIBUTES      VARCHAR2(500), /* 11 Aug 2023 Increase length of this from 255 to 1000 to avoid database conversion errors where this might have been extended */
    CONSTRAINT SBCHPANL_PK PRIMARY KEY (CHART_ID, PANEL_NO),
    CONSTRAINT SBCHPANL_FK1 FOREIGN KEY (CHART_ID) REFERENCES SBCHARTS(CHART_ID)
); 

CREATE TABLE SBGS (
    WELL_ID         NUMBER(11,0),
    DEPTH           NUMBER NOT NULL,
    GRAIN_SIZE      NUMBER  NOT NULL,
    CONSTRAINT SBGS_PK PRIMARY KEY (WELL_ID, DEPTH),
    CONSTRAINT SBGS_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
); 

CREATE TABLE SBILITH (
    WELL_ID         NUMBER(11,0),
    TOP_DEPTH       NUMBER NOT NULL,
    BASE_DEPTH      NUMBER,
    LITH_ID         NUMBER(5,0)    NOT NULL,
    COMMENTS         VARCHAR2(255),
    CONSTRAINT SBILITH_PK PRIMARY KEY (WELL_ID, TOP_DEPTH),
    CONSTRAINT SBILITH_FK1 FOREIGN KEY (LITH_ID) REFERENCES LITHDESC(LITH_ID),
    CONSTRAINT SBILITH_FK2 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
); 

/* Obsolete for v3 except for legacy charts */
CREATE TABLE SBTEMPLT (
    TEMP_ID         NUMBER(5,0),
    GLOBAL          VARCHAR2(1) CHECK (GLOBAL IN ('Y','N')),
    OWNER           VARCHAR2(5),
    NAME            VARCHAR2(50),
    CONSTRAINT SBTEMPLT_PK PRIMARY KEY (TEMP_ID)
); 
CREATE INDEX NAME_IDX2 ON SBTEMPLT(NAME);


/* Obsolete for v3 except for legacy charts */
CREATE TABLE SBLOGTPL (
    TEMP_ID         NUMBER(5,0),
    PANEL_NO        NUMBER(5,0),
    TRACE_NO        NUMBER(5,0),
    ATTRIBUTES      VARCHAR2(200),
    CONSTRAINT SBLOGTPL_PK PRIMARY KEY (TEMP_ID, PANEL_NO, TRACE_NO),
    CONSTRAINT SBLOGTPL_FK1 FOREIGN KEY (TEMP_ID) REFERENCES SBTEMPLT(TEMP_ID)
);

/* Obsolete for v3 except for legacy charts */
CREATE TABLE SBLOGTRC (
    CHART_ID        NUMBER(11,0),
    PANEL_NO        NUMBER(5,0),
    TRACE_NO        NUMBER(5,0),
    ATTRIBUTES      VARCHAR2(200),
    CONSTRAINT SBLOGTRC_PK PRIMARY KEY (CHART_ID, PANEL_NO, TRACE_NO),
    CONSTRAINT SBLOGTRC_FK1 FOREIGN KEY (CHART_ID) REFERENCES SBCHARTS(CHART_ID)
); 

CREATE TABLE SBLTHSCH (
    LTHSCH_ID       NUMBER(5,0),
    DESCR           VARCHAR2(40)    NOT NULL,
    CONSTRAINT SBLTHSCH_PK PRIMARY KEY (LTHSCH_ID)
); 

CREATE TABLE SBLTHMBR (
    LTHSCH_ID       NUMBER(5,0),
    LITH_ID         NUMBER(5,0),
    CONSTRAINT SBLTHMBR_PK PRIMARY KEY (LTHSCH_ID, LITH_ID),
    CONSTRAINT SBLTHMBR_FK1 FOREIGN KEY (LTHSCH_ID) REFERENCES SBLTHSCH(LTHSCH_ID),
    CONSTRAINT SBLTHMBR_FK2 FOREIGN KEY (LITH_ID) REFERENCES LITHDESC(LITH_ID)
); 

CREATE TABLE SBLTHPAT (
    LITH_ID         NUMBER(5,0),
    LINE            NUMBER(5,0),
    DESCR           VARCHAR2(50)    NOT NULL,
    CONSTRAINT SBLTHPAT_PK PRIMARY KEY (LITH_ID, LINE),
    CONSTRAINT SBLTHPAT_FK1 FOREIGN KEY (LITH_ID) REFERENCES LITHDESC(LITH_ID)
); 

CREATE TABLE SBLTHTRN (
    LITH_ID         NUMBER(5,0),
    DESCR           VARCHAR2(50),
    IMPCODE         NUMBER(11,0),
    SCH_ID          NUMBER(5,0),
    CONSTRAINT SBLTHTRN_PK PRIMARY KEY (LITH_ID, DESCR, IMPCODE, SCH_ID),
    CONSTRAINT SBLTHTRN_FK1 FOREIGN KEY (LITH_ID) REFERENCES LITHDESC(LITH_ID),
    CONSTRAINT SBLTHTRN_FK2 FOREIGN KEY (SCH_ID) REFERENCES LTHTRNSCH(SCH_ID)
); 

/* Obsolete for v3 except for legacy charts */
CREATE TABLE SBPANELS (
    TEMP_ID         NUMBER(5,0),
    PANEL_NO        NUMBER(5,0),
    ATTRIBUTES      VARCHAR2(255)   NOT NULL,
    CONSTRAINT SBPANELS_PK PRIMARY KEY (TEMP_ID, PANEL_NO),
    CONSTRAINT SBPANELS_FK1 FOREIGN KEY (TEMP_ID) REFERENCES SBTEMPLT (TEMP_ID)
); 

CREATE TABLE SBQLITH (
    WELL_ID         NUMBER(11,0),
    TYPE            VARCHAR2(1),
    TOP_DEPTH       NUMBER NOT NULL,
    BASE_DEPTH      NUMBER,     /* Attribute not used but reserved for future use */
    ALIGNMENT       VARCHAR2(1),
    PLOTPOS         NUMBER,
    LITH_ID         NUMBER(5,0),
    TXT         VARCHAR2(10),
    CONSTRAINT SBQLITH_PK PRIMARY KEY (WELL_ID, TYPE, TOP_DEPTH, PLOTPOS, LITH_ID),
    CONSTRAINT SBQLITH_FK1 FOREIGN KEY (LITH_ID) REFERENCES LITHDESC(LITH_ID),
    CONSTRAINT SBQLITH_FK2 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
); 

CREATE TABLE SBSCALES (
    SCALE           NUMBER,
    CONSTRAINT SBSCALES_PK PRIMARY KEY (SCALE)
); 

CREATE TABLE SBSLITH (
    WELL_ID         NUMBER(11,0),
    SAMP_ID         NUMBER(11,0),
    LITH_ID         NUMBER(5,0),
    PERCNT          NUMBER(3,0)  CHECK (PERCNT BETWEEN 0 AND 100),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SBSLITH_PK PRIMARY KEY (WELL_ID, SAMP_ID, LITH_ID),
    CONSTRAINT SBSLITH_FK1 FOREIGN KEY (WELL_ID, SAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT SBSLITH_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SBSLITH_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SBSLITH_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE SBSSR (
    WELL_ID         NUMBER(11,0),
    SAMP_ID         NUMBER(11,0),
    RATIO           NUMBER,
    AGE             NUMBER  NOT NULL,
    AGEPLUS         NUMBER,
    AGEMINUS        NUMBER,
    AGE_BELOW       NUMBER,
    INTERP_ID       NUMBER(5,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SBSSR_PK PRIMARY KEY (WELL_ID, SAMP_ID, INTERP_ID),
    CONSTRAINT SBSSR_FK1 FOREIGN KEY (WELL_ID, SAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT SBSSR_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT SBSSR_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SBSSR_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SBSSR_FK5 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE SETMBR (
    GRPSET_ID       NUMBER(5,0),
    GRP_ID          NUMBER(5,0),
    GROUP_NO        NUMBER(5,0),
    CONSTRAINT SETMBR_PK PRIMARY KEY (GRPSET_ID, GRP_ID),
    CONSTRAINT SETMBR_FK1 FOREIGN KEY (GRPSET_ID) REFERENCES GROUPSET(GRPSET_ID),
    CONSTRAINT SETMBR_FK2 FOREIGN KEY (GRP_ID) REFERENCES TXGROUP(GRP_ID)
); 

CREATE TABLE SIPMDICT (
    CCODE           NUMBER(5,0),
    TXT             VARCHAR2(60)    NOT NULL,
    CONSTRAINT SIPMDICT_PK PRIMARY KEY (CCODE)
); 

CREATE TABLE SIPMCODE (
    SPEC_ID         NUMBER(11,0),
    CCODE           NUMBER(5,0),
    SIPM_CODE       NUMBER(11,0)   NOT NULL,
    CONSTRAINT SIPMCODE_PK PRIMARY KEY (SPEC_ID, CCODE),
    CONSTRAINT SIPMCODE_FK1 FOREIGN KEY (CCODE) REFERENCES SIPMDICT(CCODE),
    CONSTRAINT SIPMCODE_FK2 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID)
); 
CREATE INDEX SIPM_CODE_IDX1 ON SIPMCODE(SIPM_CODE);


CREATE TABLE SMPDTL (
    WELL_ID         NUMBER(11,0),
    SAMP_ID         NUMBER(11,0),
    ANALY_ID        NUMBER(11,0),
    LABEL           VARCHAR2(30),   /* new for v3.0 */
    PICKER          VARCHAR2(5),
    SOURCE          VARCHAR2(5),
    BARREN          VARCHAR2(1),
    FOV             NUMBER(5,0),
    WEIGHT          NUMBER,
    COARSE          NUMBER,
    MEDIUM          NUMBER,
    FINE            NUMBER,
    NOTES           VARCHAR2(240),
    PROXIMAL        NUMBER(2,0),
    DISTAL          NUMBER(2,0),
    IMAGE_SET_ID    NUMBER(11,0),
    URL             VARCHAR2(255),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SMPDTL_PK PRIMARY KEY (WELL_ID, SAMP_ID, ANALY_ID),
    CONSTRAINT SMPDTL_FK1 FOREIGN KEY (WELL_ID, SAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT SMPDTL_FK2 FOREIGN KEY (ANALY_ID, WELL_ID) REFERENCES ANALY_HDR(ANALY_ID, WELL_ID),
    CONSTRAINT SMPDTL_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SMPDTL_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SMPDTL_FK5 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SMPDTL_FK6 FOREIGN KEY (IMAGE_SET_ID) REFERENCES IMAGE_SET (IMAGE_SET_ID)
); 

CREATE TABLE SMPTYPES (
    TYPE_ABR        VARCHAR2(5),
    TYPE_NAME       VARCHAR2(30),
    CONSTRAINT SMPTYPES_PK PRIMARY KEY (TYPE_ABR)
); 

CREATE TABLE SPECIESTYPE (
    SPEC_TYPE_ID    NUMBER(5,0),
    DESCR           VARCHAR2(50)    NOT NULL,
    CONSTRAINT SPECIESTYPE_PK PRIMARY KEY (SPEC_TYPE_ID)
); 

CREATE TABLE SPLITID (
    USRID           NUMBER NOT NULL, 
    LABEL           VARCHAR2(30) NOT NULL, 
    ID              NUMBER NOT NULL, 
    CONSTRAINT SPLITID_PK PRIMARY KEY (USRID, LABEL, ID)
);

CREATE TABLE SURFACE (
    SURFACE_ID      NUMBER(5,0),
    SCH_ID          NUMBER(5,0),
    AGE             NUMBER,
    NAME            VARCHAR2(80),   /* Changed from 50 to 80 on 11 Mar 2014 */
    /* to add ABR for v3.1 */
    TYPE            NUMBER(1,0),
    MAGNITUDE       NUMBER(5,0),
    CONFIDENCE      NUMBER(5,0),
    /* to add comments for v3.1 */
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SURFACE_PK PRIMARY KEY (SURFACE_ID),
    CONSTRAINT SURFACE_FK1 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID),
    CONSTRAINT SURFACE_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SURFACE_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SURFACE_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX SCH_ID_IDX2 ON SURFACE(SCH_ID);

CREATE TABLE SQPICK (
    WELL_ID         NUMBER(11,0),
    INTERP_ID       NUMBER(5,0),
    SURFACE_ID      NUMBER(5,0),
    SAMP_ID         NUMBER(11,0),
    INF             VARCHAR2(80),
    BND             NUMBER(1),
    QUALIFIER       VARCHAR2(1),
    OBSERVED        VARCHAR2(1),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT SQPICK_PK PRIMARY KEY (WELL_ID, INTERP_ID, SURFACE_ID, SAMP_ID),
    CONSTRAINT SQPICK_FK1 FOREIGN KEY (WELL_ID, SAMP_ID) REFERENCES SAMPLES(WELL_ID, SAMP_ID),
    CONSTRAINT SQPICK_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT SQPICK_FK3 FOREIGN KEY (SURFACE_ID) REFERENCES SURFACE(SURFACE_ID),
    CONSTRAINT SQPICK_FK4 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SQPICK_FK5 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT SQPICK_FK6 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE CHRON (
    CHRON_ID        NUMBER(11,0) PRIMARY KEY NOT NULL,
    SCH_ID          NUMBER(5,0) NOT NULL,
    HIER            NUMBER(5,0) NOT NULL, 
    NAME            VARCHAR2(80) NOT NULL, /* Need to set this to not allow zero length! */
    U_AGE           NUMBER,          /* This changed from NUMBER(11,3) to NUMBER 13 Jul 2023 */
    L_AGE           NUMBER,
    POLARITY        VARCHAR2(1),
    CREATED         DATE, 
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CHRON_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHRON_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHRON_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHRON_FK4 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH(SCH_ID)
);

CREATE TABLE SYNSCH (
    SCH_ID          NUMBER(5,0),
    NAME            VARCHAR2(30)    NOT NULL,
    CONSTRAINT SYNSCH_PK PRIMARY KEY (SCH_ID)
); 

CREATE TABLE SYNONYMY (
    SCH_ID          NUMBER(5,0),
    SPEC_ID         NUMBER(11,0),
    PREF            NUMBER(11,0)   NOT NULL,
    CONSTRAINT SYNONYMY_PK PRIMARY KEY (SCH_ID, SPEC_ID),
    CONSTRAINT SYNONYMY_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT SYNONYMY_FK2 FOREIGN KEY (PREF) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT SYNONYMY_FK3 FOREIGN KEY (SCH_ID) REFERENCES SYNSCH(SCH_ID)
); 
CREATE INDEX PREF_IDX1 ON SYNONYMY(PREF);


CREATE TABLE TAXONOCC (
    WELL_ID         NUMBER(11,0),
    SAMP_ID         NUMBER(11,0),
    ANALY_ID        NUMBER(11,0),
    SPEC_ID         NUMBER(11,0),
    SITUATION       VARCHAR2(1),    CHECK (SITUATION IN ('I', 'T', 'R', 'C', 'N')), /* Added N for contamination 11 Oct 2023 */
    IDENT_TYPE      VARCHAR2(1),    CHECK (IDENT_TYPE IN ('P', '?')),
    SPEC_TYPE_ID    NUMBER(5,0),   /* Linked to SPECIESTYPE  table and replaces FORM and GROWTH. Default is zero for no sub type, or data not sub typed.*/
    MARKER          VARCHAR2(1),
    ABUND           VARCHAR2(4),    /* increased to 4 chars in v2.1 */
    COARSE          NUMBER,
    MEDIUM          NUMBER,
    FINE            NUMBER,
/*    CAVED           VARCHAR2(1),  Removed for v3.0, caved occs now part of SITUATION field */
    PRESERV         VARCHAR2(20),
    COLOUR          VARCHAR2(20),
    COMMENTS        VARCHAR2(255),
    IMAGE_SET_ID    NUMBER(11,0),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT TAXONOCC_PK PRIMARY KEY (WELL_ID, SAMP_ID, ANALY_ID, SPEC_ID, SITUATION, IDENT_TYPE, SPEC_TYPE_ID),
    CONSTRAINT TAXONOCC_FK1 FOREIGN KEY (WELL_ID, SAMP_ID, ANALY_ID) REFERENCES SMPDTL(WELL_ID, SAMP_ID, ANALY_ID),
    CONSTRAINT TAXONOCC_FK2 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT TAXONOCC_FK3 FOREIGN KEY (SPEC_TYPE_ID) REFERENCES SPECIESTYPE(SPEC_TYPE_ID),
    CONSTRAINT TAXONOCC_FK4 FOREIGN KEY (IMAGE_SET_ID) REFERENCES IMAGE_SET(IMAGE_SET_ID),
    CONSTRAINT TAXONOCC_FK5 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT TAXONOCC_FK6 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT TAXONOCC_FK7 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 
CREATE INDEX IMAGE_SET_ID_IDX1 ON TAXONOCC(IMAGE_SET_ID);
CREATE INDEX TAXONOCC_IDX ON TAXONOCC(SPEC_ID);


CREATE TABLE TVDHDR (
    WELL_ID         NUMBER(11,0),
    ISPLAN          NUMBER(1,0)     NOT NULL CHECK (ISPLAN BETWEEN 0 AND 1),
    DATUM           NUMBER(1,0)    CHECK (DATUM BETWEEN 0 AND 3),    /*0 MEANS NO INFO*/
    COMMENTS        CLOB,
    CONSTRAINT TVDHDR_PK PRIMARY KEY (WELL_ID, ISPLAN),
    CONSTRAINT TVDHDR_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
);

/* TXDEPTH table exists in previous versions but not implemented.
 * Modified for v3.0 to allow water depth range analysis and derivation of IPS type curves
 * Depth values replaced by bathymetric zones related to an env scheme.
 */
CREATE TABLE TXDEPTH (
    SPEC_ID         NUMBER(11,0),
    ENVSCH_ID       NUMBER(5,0),
    UPPER_ENV       NUMBER(3,0),
    LOWER_ENV       NUMBER(3,0),
    CONFIDENCE      NUMBER(5,0) DEFAULT 0 NOT NULL,
    COMMENTS        VARCHAR2(255),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),      
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),      
    CONSTRAINT TXDEPTH_PK PRIMARY KEY (SPEC_ID, ENVSCH_ID),
    CONSTRAINT TXDEPTH_FK1 FOREIGN KEY (ENVSCH_ID) REFERENCES ENV_SCH(ENVSCH_ID)
);

CREATE TABLE TXIMAGE (
    SPEC_ID         NUMBER(11,0),
    IMAGE_SET_ID    NUMBER(11,0),    /* Links to IMAGE_SET */
    TYPE        VARCHAR2(1)    CHECK (TYPE IN ('Y', 'N')),     /*Flag Y/N to indicate whether set is a type specimen (holotype/paratype) */
    CONSTRAINT TXIMAGE_PK PRIMARY KEY (SPEC_ID, IMAGE_SET_ID),
    CONSTRAINT TXIMAGE_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT TXIMAGE_FK2 FOREIGN KEY (IMAGE_SET_ID) REFERENCES IMAGE_SET(IMAGE_SET_ID)
); 

CREATE TABLE TXLOAD (
    SOURCE_ID       VARCHAR2(8),
    SPEC_ID         NUMBER(11,0),
    TXT             VARCHAR2(150),
    OCC_TYPE        VARCHAR2(20),
    CONSTRAINT TXLOAD_PK PRIMARY KEY (SOURCE_ID, TXT, OCC_TYPE),
    CONSTRAINT TXLOAD_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES(SPEC_ID),
    CONSTRAINT TXLOAD_FK2 FOREIGN KEY (SOURCE_ID) REFERENCES SOURCE(SOURCE_ID)
); 
CREATE INDEX TXLOAD_IDX ON TXLOAD(SPEC_ID);


/* Defunct in v3.0, now part of SPECIES table
TABLE TXNOTES (
    SPEC_ID         NUMBER(11,0),
    NOTES           VARCHAR2(240)   NOT NULL,
    CONSTRAINT TXNOTES_PK PRIMARY KEY (SPEC_ID)
); 

TABLE TXREFS (
    SPEC_ID         NUMBER(11,0),
    REFERENCE       VARCHAR2(240)   NOT NULL,
    CONSTRAINT TXREFS_PK PRIMARY KEY (SPEC_ID)
); 
*/

CREATE TABLE WELLSMARK (
    WELL_ID         NUMBER(11,0),
    DEPTH           NUMBER NOT NULL,
    NAME            VARCHAR2(20)    NOT NULL,
    COLOUR          VARCHAR2(9),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT WELLSMARK_PK PRIMARY KEY (WELL_ID, DEPTH),
    CONSTRAINT WELLSMARK_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT WELLSMARK_FK2 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT WELLSMARK_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT WELLSMARK_FK4 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID)
); 

CREATE TABLE WELLTVD (
    WELL_ID         NUMBER(11,0),
    ISPLAN          NUMBER(1,0) NOT NULL CHECK (ISPLAN BETWEEN 0 AND 1),
    DDEPTH          NUMBER NOT NULL,
    TVDEPTH         NUMBER NOT NULL,
    INCLINATION     NUMBER,
    AZIMUTH         NUMBER,
    INCRVS          NUMBER,
    NORTH           NUMBER,
    EAST            NUMBER,
    CONSTRAINT WELLTVD_PK PRIMARY KEY (WELL_ID, ISPLAN, DDEPTH),
    CONSTRAINT WELLTVD_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
); 

CREATE TABLE WELLTWT (
    WELL_ID         NUMBER(11,0),
    DDEPTH          NUMBER NOT NULL,
    TWT             NUMBER NOT NULL,
    CONSTRAINT WELLTWT_PK PRIMARY KEY (WELL_ID, DDEPTH),
    CONSTRAINT WELLTWT_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID)
); 

CREATE TABLE CHART (    
    CHART_ID        NUMBER(5,0) PRIMARY KEY NOT NULL, 
    DESCR           VARCHAR2(255), 
    COMMENTS        VARCHAR2(255), 
    PROPS           VARCHAR2(1000), /* increased from 255 to 1000 for v3.0 to accommodate long titles */ 
    PROJ_ID         NUMBER(5,0), 
    WELLIST_ID      NUMBER(5,0),    /* the chart has a default well list tag, but is not restricted by it */
    SCH_ID          NUMBER(5,0),
    HEADER          VARCHAR2(255),
    CHT_KEY         VARCHAR2(255),
    ACM             NUMBER(5,0),
    CREATED         DATE, 
    CREATOR         NUMBER(5,0), 
    MODIFIED        DATE, 
    MODIFIER        NUMBER(5,0), 
    UPDATED         TIMESTAMP, 
    UPDATER         NUMBER(5,0),
    CONSTRAINT CHART_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHART_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHART_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHART_FK4 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID),
    CONSTRAINT CHART_FK5 FOREIGN KEY (WELLIST_ID,PROJ_ID) REFERENCES WELLIST(WELLIST_ID,PROJ_ID),
    CONSTRAINT CHART_FK6 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID)
);

CREATE TABLE CHART_ACL (
    CHART_ID        NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT CHART_ACL_PK PRIMARY KEY (CHART_ID, USER_ID),
    CONSTRAINT CHART_ACL_FK1 FOREIGN KEY (CHART_ID) REFERENCES CHART (CHART_ID),
    CONSTRAINT CHART_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE CHARTCORR (
    CORRSCH_ID      NUMBER(5,0) PRIMARY KEY NOT NULL, 
    CORR_TYPE       VARCHAR2(15),
    DESCR           VARCHAR2(255), 
    COMMENTS        VARCHAR2(255), 
    PROJ_ID         NUMBER(5,0),
    ACM             NUMBER(5,0),
    CREATED         DATE, 
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CHARTCORR_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHARTCORR_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHARTCORR_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHARTCORR_FK4 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID)
);

CREATE TABLE CHTBLOCK (
    BLOCK_ID        NUMBER(5,0) PRIMARY KEY NOT NULL, 
    TYPE            VARCHAR2(10) NOT NULL,
    DESCR           VARCHAR2(255), 
    COMMENTS        VARCHAR2(255), 
    PROJ_ID         NUMBER(5,0),
    WELL_ID         NUMBER(11,0),
    PARENT_ID       NUMBER(5,0),
    ACM             NUMBER(5,0),
    CREATED         DATE, 
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CHTBLOCK_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHTBLOCK_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHTBLOCK_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHTBLOCK_FK4 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID),
    CONSTRAINT CHTBLOCK_FK5 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT CHTBLOCK_FK6 FOREIGN KEY (PARENT_ID) REFERENCES CHTBLOCK(BLOCK_ID)
);

CREATE TABLE CHTBLOCK_ACL (
    BLOCK_ID        NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT CHTBLOCK_ACL_PK PRIMARY KEY (BLOCK_ID, USER_ID),
    CONSTRAINT CHTBLOCK_ACL_FK1 FOREIGN KEY (BLOCK_ID) REFERENCES CHTBLOCK (BLOCK_ID),
    CONSTRAINT CHTBLOCK_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE SHAPESTORE (
    ID              NUMBER(11,0)    NOT NULL,
    TYPE            VARCHAR2(20)    NOT NULL,
    NAME            VARCHAR2(255)   NOT NULL,
    URL             VARCHAR2(1000)  NOT NULL,
    LONGFIRST       BIT NOT NULL,
    CONSTRAINT SHAPESTORE_PK PRIMARY KEY (ID)
);

CREATE TABLE SHAPESTORE_DATA (
    SHAPESTORE_ID   NUMBER(11,0) NOT NULL,
    FILENAME        VARCHAR2(255) NOT NULL,
    SHAPE_DATA      BLOB NULL,
    CONSTRAINT SHAPESTOREDATA_PK PRIMARY KEY (SHAPESTORE_ID, FILENAME),
    CONSTRAINT SHAPESTOREDATA_FK FOREIGN KEY (SHAPESTORE_ID) REFERENCES SHAPESTORE(ID)
);

CREATE TABLE CHTPANL (
    PANEL_ID        NUMBER(11,0) PRIMARY KEY NOT NULL,
    TYPE            VARCHAR2(15) NOT NULL,
    DESCR           VARCHAR2(255), 
    COMMENTS        VARCHAR2(255),
    ISDEFAULT       VARCHAR2(1),
    ISVISIBLE       VARCHAR2(1) NOT NULL,
    PROP            VARCHAR2(1000), /* Increased from 255 on 28 Feb 2023 */
    PROJ_ID         NUMBER(5,0),
    WELLIST_ID      NUMBER(5,0),
    SCH_ID          NUMBER(5,0),
    STD_ID          NUMBER(5,0),
    ENVSCH_ID       NUMBER(5,0),
    SYNSCH_ID       NUMBER(5,0),
    GRPSET_ID       NUMBER(5,0),
    AGECURVE_ID     NUMBER(5,0),
    SHAPE_ID        NUMBER(11,0),
    ACM             NUMBER(5,0),
    CREATED         DATE, 
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CHTPANL_CHK CHECK (ISVISIBLE IN ('Y', 'N')),
    CONSTRAINT CHTPANL_FK1 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHTPANL_FK2 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHTPANL_FK3 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHTPANL_FK4 FOREIGN KEY (PROJ_ID) REFERENCES PROJECT(PROJ_ID),
    CONSTRAINT CHTPANL_FKWL FOREIGN KEY (WELLIST_ID,PROJ_ID) REFERENCES WELLIST(WELLIST_ID,PROJ_ID),
    CONSTRAINT CHTPANL_FK5 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID),
    CONSTRAINT CHTPANL_FK6 FOREIGN KEY (STD_ID) REFERENCES CMPSTD (STD_ID),
    CONSTRAINT CHTPANL_FK7 FOREIGN KEY (ENVSCH_ID) REFERENCES  ENV_SCH(ENVSCH_ID),
    CONSTRAINT CHTPANL_FK8 FOREIGN KEY (SYNSCH_ID) REFERENCES SYNSCH(SCH_ID),
    CONSTRAINT CHTPANL_FK9 FOREIGN KEY (GRPSET_ID) REFERENCES GROUPSET(GRPSET_ID),
    CONSTRAINT CHTPANL_FK10 FOREIGN KEY (AGECURVE_ID) REFERENCES AGE_CURVE(CURVE_ID),
    CONSTRAINT CHTPANL_SHAPE_ID_FKEY FOREIGN KEY (SHAPE_ID) REFERENCES SHAPESTORE(ID)
);

CREATE TABLE CHTPANL_ACL (
    PANEL_ID        NUMBER(11,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT CHTPANL_ACL_PK PRIMARY KEY (PANEL_ID, USER_ID),
    CONSTRAINT CHTPANL_ACL_FK1 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID),
    CONSTRAINT CHTPANL_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE CHTBLOCKMBR (
    BLOCK_ID        NUMBER(5,0) NOT NULL,
    PANEL_ID        NUMBER(11,0) NOT NULL,
    PANEL_NO        NUMBER(5,0) NOT NULL,
    INTERP_ID       NUMBER(5,0),
    CAPTION         VARCHAR2(255), 
    SUBCAPTION      VARCHAR2(255),
    SCH_ID          NUMBER(5,0),
    OVERPLOT        VARCHAR2(6),
    CONSTRAINT CHTBLOCKMBR_PK PRIMARY KEY (BLOCK_ID, PANEL_NO),
    CONSTRAINT OVP_CHK CHECK (OVERPLOT IN ('LEFT', 'CENTRE', 'RIGHT')),    /* Indicates that panel can overplot previous panel */
    CONSTRAINT CHTBLOCKMBR_FK1 FOREIGN KEY (BLOCK_ID) REFERENCES CHTBLOCK (BLOCK_ID),
    CONSTRAINT CHTBLOCKMBR_FK2 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID),
    CONSTRAINT CHTBLOCKMBR_FK3 FOREIGN KEY (INTERP_ID) REFERENCES INTERP (INTERP_ID),
    CONSTRAINT CHTBLOCKMBR_FK4 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID)
);

CREATE TABLE CHTCORROCC (
    CHART_ID        NUMBER(5,0) NOT NULL, 
    CORRSCH_ID      NUMBER(5,0) NOT NULL,
    INTERP_ID       NUMBER(5,0) NOT NULL,
    PROP            VARCHAR2(255), 
    STYLE           VARCHAR2(255),
    DATUM           VARCHAR2(50),
    CONSTRAINT CHTCORROCC_PK PRIMARY KEY (CHART_ID, CORRSCH_ID, INTERP_ID),
    CONSTRAINT CHTCORROCC_FK1 FOREIGN KEY (INTERP_ID) REFERENCES INTERP (INTERP_ID),
    CONSTRAINT CHTCORROCC_FK2 FOREIGN KEY (CHART_ID) REFERENCES CHART (CHART_ID),
    CONSTRAINT CHTCORROCC_FK3 FOREIGN KEY (CORRSCH_ID) REFERENCES CHARTCORR (CORRSCH_ID)
);

CREATE TABLE CHTCORROCC_STD (
    CHART_ID        NUMBER(5,0) NOT NULL, 
    TYPE            VARCHAR2(10) NOT NULL,
    SCOPE           VARCHAR2(10) NOT NULL,
    DATA_TYPES      VARCHAR2(100) NOT NULL,
    INTERP_ID       NUMBER(5,0) NOT NULL,
    PROP            VARCHAR2(255), 
    STYLE           VARCHAR2(255),
    CONSTRAINT CHTCORROCC_STD_PK PRIMARY KEY (CHART_ID, TYPE, SCOPE, DATA_TYPES, INTERP_ID),
    CONSTRAINT CHTCORROCC_STD_FK1 FOREIGN KEY (INTERP_ID) REFERENCES INTERP (INTERP_ID),
    CONSTRAINT CHTCORROCC_STD_FK2 FOREIGN KEY (CHART_ID) REFERENCES CHART (CHART_ID)
);

CREATE TABLE CHARTCORR_ACL (
    CORRSCH_ID      NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL,
    CONSTRAINT CHARTCORR_ACL_PK PRIMARY KEY (CORRSCH_ID, USER_ID),
    CONSTRAINT CHARTCORR_ACL_FK1 FOREIGN KEY (CORRSCH_ID) REFERENCES CHARTCORR (CORRSCH_ID),
    CONSTRAINT CHARTCORR_ACL_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE CHTCORROCC_STD_UNIT (
    CHART_ID        NUMBER(5,0) NOT NULL, 
    TYPE            VARCHAR2(10) NOT NULL,
    SCOPE           VARCHAR2(10) NOT NULL,
    DATA_TYPES      VARCHAR2(100) NOT NULL,
    INTERP_ID       NUMBER(5,0) NOT NULL,
    SCH_ID          NUMBER(5,0) NOT NULL,
    HIER            NUMBER(5,0) NOT NULL,
    PROP            VARCHAR2(255), 
    STYLE           VARCHAR2(255),
    CONSTRAINT CHTCORROCC_STD_UNIT_PK PRIMARY KEY (CHART_ID, TYPE, SCOPE, DATA_TYPES, INTERP_ID, SCH_ID, HIER),
    CONSTRAINT CHTCORROCC_STD_UNIT_FK1 FOREIGN KEY (INTERP_ID) REFERENCES INTERP (INTERP_ID),
    CONSTRAINT CHTCORROCC_STD_UNIT_FK2 FOREIGN KEY (CHART_ID) REFERENCES CHART (CHART_ID),
    CONSTRAINT CHTCORROCC_STD_UNIT_FK3 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID)
);

CREATE TABLE CHTLN_EV (
    CORRSCH_ID      NUMBER(5,0) NOT NULL, 
    EV_ID           NUMBER(11,0) NOT NULL,
    EV_TYPE         VARCHAR2(10) NOT NULL,
    STYLE           VARCHAR2(255),
    CONSTRAINT CHTLN_EV_PK PRIMARY KEY (CORRSCH_ID, EV_ID, EV_TYPE),
    CONSTRAINT CHTLN_EV_FK1 FOREIGN KEY (CORRSCH_ID) REFERENCES CHARTCORR (CORRSCH_ID),
    CONSTRAINT CHTLN_EV_FK2 FOREIGN KEY (EV_ID) REFERENCES EVENTDIC (EV_ID)
);

CREATE TABLE CHTLN_SURFACE (
    CORRSCH_ID      NUMBER(5,0) NOT NULL, 
    SURFACE_ID      NUMBER(5,0) NOT NULL,
    STYLE           VARCHAR2(255),
    CONSTRAINT CHTLN_SURFACE_PK PRIMARY KEY (CORRSCH_ID, SURFACE_ID),
    CONSTRAINT CHTLN_SURFACE_FK1 FOREIGN KEY (CORRSCH_ID) REFERENCES CHARTCORR (CORRSCH_ID),
    CONSTRAINT CHTLN_SURFACE_FK2 FOREIGN KEY (SURFACE_ID) REFERENCES SURFACE (SURFACE_ID)
);

CREATE TABLE CHTLN_UNIT (
    CORRSCH_ID      NUMBER(5,0) NOT NULL, 
    IGD_ID          NUMBER(11,0) NOT NULL,
    BND_TYPE        VARCHAR2(4) NOT NULL,
    STYLE           VARCHAR2(255),
    CONSTRAINT CHTLN_UNIT_PK PRIMARY KEY (CORRSCH_ID, IGD_ID, BND_TYPE),
    CONSTRAINT CHTLN_UNIT_FK1 FOREIGN KEY (CORRSCH_ID) REFERENCES CHARTCORR (CORRSCH_ID),
    CONSTRAINT CHTLN_UNIT_FK2 FOREIGN KEY (IGD_ID) REFERENCES IGD_DICT (IGD_ID)
);

CREATE TABLE CHTMBR (
    CHART_ID        NUMBER(5,0) NOT NULL, 
    BLOCK_ID        NUMBER(5,0) NOT NULL,
    BLOCK_NO        NUMBER(5,0) NOT NULL,
    WELL_ID         NUMBER(11,0),
    WELLIST_ID      NUMBER(5,0),    /* Changed from PROJ_ID for v3.0 */
    INTERP_ID       NUMBER(5,0),
    PROPS           VARCHAR2(255), 
    CAPTION         VARCHAR2(255), 
    CONSTRAINT CHTMBR_PK PRIMARY KEY (CHART_ID, BLOCK_NO),
    CONSTRAINT CHTMBR_FK1 FOREIGN KEY (CHART_ID) REFERENCES CHART (CHART_ID),
    CONSTRAINT CHTMBR_FK2 FOREIGN KEY (BLOCK_ID) REFERENCES CHTBLOCK (BLOCK_ID),
    CONSTRAINT CHTMBR_FK3 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID),
    CONSTRAINT CHTMBR_FK4 FOREIGN KEY (WELLIST_ID) REFERENCES WELLIST (WELLIST_ID),
    CONSTRAINT CHTMBR_FK5 FOREIGN KEY (INTERP_ID) REFERENCES INTERP (INTERP_ID)
);

CREATE TABLE CHTPANL_BLKMBR (
    PANEL_ID        NUMBER(11,0) NOT NULL, 
    BLOCK_ID        NUMBER(5,0) NOT NULL,
    CONSTRAINT CHTPANL_BLKMBR_PK PRIMARY KEY (PANEL_ID, BLOCK_ID),
    CONSTRAINT CHTPANL_BLKMBR_FK1 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID),
    CONSTRAINT CHTPANL_BLKMBR_FK2 FOREIGN KEY (BLOCK_ID) REFERENCES CHTBLOCK (BLOCK_ID)
);

CREATE TABLE CHTPANLMBR (
    PANEL_ID        NUMBER(11,0) NOT NULL, 
    PANEL_NO        NUMBER(5,0) NOT NULL, 
    TYPE            VARCHAR2(10) NOT NULL,
    OVERPLOT        VARCHAR2(6),
    CAPTION         VARCHAR2(255), 
    PROP            VARCHAR2(255), 
    SPEC_ID         NUMBER(11,0),
    GRP_ID          NUMBER(5,0),
    GRPSET_ID       NUMBER(5,0),
    CAT_MNEM        VARCHAR2(5),
    ISDEFAULT       NUMBER(1),
    FOREACH         NUMBER(1),
    HIGHLIGHT_GRP_ID NUMBER(5,0),
    EXCLUDE_GRP_ID NUMBER(5,0), /* Added 3 March */
    STD_ID          NUMBER(5,0),
    SCH_ID          NUMBER(5,0),
    CONSTRAINT CHTPANLMBR_PK PRIMARY KEY (PANEL_ID, PANEL_NO),
    CONSTRAINT CHTPANLMBR_FK1 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID),
    CONSTRAINT CHTPANLMBR_FK2 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES (SPEC_ID),
    CONSTRAINT CHTPANLMBR_FK3 FOREIGN KEY (GRP_ID) REFERENCES TXGROUP (GRP_ID),
    CONSTRAINT CHTPANLMBR_FK4 FOREIGN KEY (GRPSET_ID) REFERENCES GROUPSET (GRPSET_ID),
    CONSTRAINT CHTPANLMBR_FK5 FOREIGN KEY (CAT_MNEM) REFERENCES CATEGORY (CAT_MNEM),
    CONSTRAINT CHTPANLMBR_FK6 FOREIGN KEY (HIGHLIGHT_GRP_ID) REFERENCES TXGROUP (GRP_ID),
    CONSTRAINT CHTPANLMBR_FK7 FOREIGN KEY (EXCLUDE_GRP_ID) REFERENCES TXGROUP (GRP_ID),
    CONSTRAINT CHTPANLMBR_FK8 FOREIGN KEY (STD_ID) REFERENCES CMPSTD (STD_ID),
    CONSTRAINT CHTPANLMBR_FK9 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID),
    CONSTRAINT OVP_CHK1 CHECK (OVERPLOT IN ('LEFT', 'CENTRE', 'RIGHT'))    /* as above but for inner panel members */
);

CREATE TABLE CHTLOG (
    PANEL_ID        NUMBER(11,0) NOT NULL, 
    ABR             VARCHAR2(10) NOT NULL,
    PROP            VARCHAR2(255), 
    CONSTRAINT CHTLOG_PK PRIMARY KEY (PANEL_ID, ABR),
    CONSTRAINT CHTLOG_FK1 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID)
);

CREATE TABLE CHTPREF_BLK (
    BLOCK_ID        NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL, 
    WELL_ID         NUMBER(11,0) NOT NULL, 
    B_PROP          VARCHAR2(255), 
    C_PROP          VARCHAR2(255), 
    HEADER          VARCHAR2(255),
    CHT_KEY         VARCHAR2(255),
    SCH_ID          NUMBER(5,0),
    CONSTRAINT CHTPREF_BLK_FK1 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID),
    CONSTRAINT CHTPREF_BLK_PK PRIMARY KEY (BLOCK_ID, USER_ID, WELL_ID),
    CONSTRAINT CHTPREF_BLK_FK2 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID),
    CONSTRAINT CHTPREF_BLK_FK3 FOREIGN KEY (BLOCK_ID) REFERENCES CHTBLOCK (BLOCK_ID),
    CONSTRAINT CHTPREF_BLK_FK4 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID)
);

CREATE TABLE CHTPREF_CHT (
    CHART_ID        NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL, 
    WELL_ID         NUMBER(11,0) NOT NULL,
    B_PROP          VARCHAR2(255), 
    CONSTRAINT CHTPREF_CHT_PK PRIMARY KEY (CHART_ID, USER_ID, WELL_ID),
    CONSTRAINT CHTPREF_CHT_FK1 FOREIGN KEY (CHART_ID) REFERENCES CHART (CHART_ID),
    CONSTRAINT CHTPREF_CHT_FK2 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID),
    CONSTRAINT CHTPREF_CHT_FK3 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID)
);

CREATE TABLE CHTPREF_SCH (
    BLOCK_ID        NUMBER(5,0) NOT NULL,
    USER_ID         NUMBER(5,0) NOT NULL, 
    WELL_ID         NUMBER(11,0), 
    WELLIST_ID      NUMBER(5,0), /*Changed from project for v3*/
    INTERP_ID       NUMBER(5,0),
    SCH_ID          NUMBER(5,0),
    B_PROP          VARCHAR2(255), 
    C_PROP          VARCHAR2(255), 
    HEADER          VARCHAR2(255),
    CHT_KEY         VARCHAR2(255),
    LOCATION        VARCHAR2(10) NOT NULL,
    CONSTRAINT CHTPREF_SCH_PK PRIMARY KEY (BLOCK_ID, USER_ID, LOCATION),
    CONSTRAINT CHTPREF_SCH_FK1 FOREIGN KEY (USER_ID) REFERENCES USERDEF (USER_ID),
    CONSTRAINT CHTPREF_SCH_FK2 FOREIGN KEY (WELLIST_ID) REFERENCES WELLIST(WELLIST_ID),
    CONSTRAINT CHTPREF_SCH_FK3 FOREIGN KEY (WELL_ID) REFERENCES WELLS (WELL_ID),
    CONSTRAINT CHTPREF_SCH_FK4 FOREIGN KEY (BLOCK_ID) REFERENCES CHTBLOCK (BLOCK_ID),
    CONSTRAINT CHTPREF_SCH_FK5 FOREIGN KEY (SCH_ID) REFERENCES IGD_SCH (SCH_ID),
    CONSTRAINT CHTPREF_SCH_FK6 FOREIGN KEY (INTERP_ID) REFERENCES INTERP (INTERP_ID)
);

/* CHTPROJ table dropped for v3.0 */

CREATE TABLE CHTANALYST (
    PANEL_ID        NUMBER(11,0) NOT NULL, 
    ANALYST_ID      NUMBER(5,0) NOT NULL,
    CONSTRAINT CHTANALYST_PK PRIMARY KEY (PANEL_ID, ANALYST_ID),
    CONSTRAINT CHTANALYST_FK1 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID),
    CONSTRAINT CHTANALYST_FK2 FOREIGN KEY (ANALYST_ID) REFERENCES USERDEF (USER_ID)
);

CREATE TABLE CHTSPECTYPE (
    PANEL_ID        NUMBER(11,0) NOT NULL, 
    PANEL_NO        NUMBER(5,0) NOT NULL, 
    SPEC_TYPE_ID    NUMBER(5,0) NOT NULL, 
    CONSTRAINT CHTSPECTYPE_PK PRIMARY KEY (PANEL_ID, PANEL_NO, SPEC_TYPE_ID),
    CONSTRAINT CHTSPECTYPE_FK1 FOREIGN KEY (PANEL_ID, PANEL_NO) REFERENCES CHTPANLMBR (PANEL_ID, PANEL_NO),
    CONSTRAINT CHTSPECTYPE_FK2 FOREIGN KEY (SPEC_TYPE_ID) REFERENCES SPECIESTYPE(SPEC_TYPE_ID)
);

/* Defunct in v3.0 now part of SPECIES
TABLE TXURL (
    SPEC_ID         NUMBER(11,0) PRIMARY KEY NOT NULL,
    URL VARCHAR2(255) NOT NULL,
    CONSTRAINT TXURL_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES (SPEC_ID)
);
*/

CREATE TABLE CHTLN_UNIT_LSTRAT (
    CORRSCH_ID      NUMBER(5,0) NOT NULL, 
    UNIT_ID         NUMBER(11,0) NOT NULL,
    BND_TYPE        VARCHAR2(4) NOT NULL,
    STYLE           VARCHAR2(255),
    CONSTRAINT CHTLN_UNIT_LSTRAT_PK PRIMARY KEY (CORRSCH_ID, UNIT_ID, BND_TYPE),
    CONSTRAINT CHTLN_UNIT_LSTRAT_FK1 FOREIGN KEY (CORRSCH_ID) REFERENCES CHARTCORR (CORRSCH_ID),
    CONSTRAINT CHTLN_UNIT_LSTRAT_FK2 FOREIGN KEY (UNIT_ID) REFERENCES IGD_DICT_LSTRAT (UNIT_ID)
);


CREATE TABLE DBVERSION (
    VERSION         VARCHAR2(10),
    DESCR           VARCHAR2(80),
    MODIFIED        DATE
);

/* new for v3.0 to store PDF documents in database */
CREATE TABLE WELL_DOCS (
    DOC_ID          NUMBER(11,0)    NOT NULL,
    WELL_ID         NUMBER(11,0)    NOT NULL,
    INTERP_ID       NUMBER(5,0),    /* Can be null if not linked to any version */
    DESCRIP         VARCHAR2(255)   NOT NULL,
    DOC_PATH        VARCHAR2(255),
    DOC             BLOB            NOT NULL,
    CREATED         DATE,           /* not all the audit fields are used here, just created/creator */
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT DOCS_PK PRIMARY KEY (DOC_ID),
    CONSTRAINT DOCS_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT DOCS_FK2 FOREIGN KEY (INTERP_ID) REFERENCES INTERP(INTERP_ID),
    CONSTRAINT DOCS_FK3 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT DOCS_FK4 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT DOCS_FK5 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID)
);

/* this table is for references to a document outside the primary well_id reference e.g. if a report covers several wells*/
CREATE TABLE DOC_REFS (
    DOC_ID      NUMBER(11,0),
    WELL_ID     NUMBER(11,0),
    CONSTRAINT DOC_REFS_PK PRIMARY KEY (DOC_ID, WELL_ID),
    CONSTRAINT DOC_REFS_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT DOC_REFS_FK2 FOREIGN KEY (DOC_ID) REFERENCES WELL_DOCS(DOC_ID)
);

CREATE TABLE CHTTAG (
    TAG_ID       NUMBER(11,0) NOT NULL,
    CHART_ID     NUMBER(5,0) NOT NULL,  /* This needs to agree with chart.chart_id */
    BLOCK_NO     NUMBER(5,0),
    TAG_ORDER    NUMBER(11,0),
    POS_X        NUMBER NOT NULL,
    POS_Y        NUMBER NOT NULL,
    WIDTH        NUMBER NOT NULL,
    HEIGHT       NUMBER NOT NULL,
    POS_DEPTH    NUMBER,
    POS_LAT      NUMBER,
    POS_LON      NUMBER,
    TARGET_X     NUMBER,
    TARGET_Y     NUMBER,
    TARGET_DEPTH NUMBER,
    TARGET_LAT   NUMBER,
    TARGET_LON   NUMBER,
    TEXT         VARCHAR2(1400),
    PROP         VARCHAR2(1400),
    CONSTRAINT CHTTAG_PK PRIMARY KEY (TAG_ID),
    CONSTRAINT CHTTAG_FK1 FOREIGN KEY (CHART_ID) REFERENCES CHART(CHART_ID),
    CONSTRAINT CHTTAG_FK2 FOREIGN KEY (CHART_ID, BLOCK_NO) REFERENCES CHTMBR(CHART_ID, BLOCK_NO)
);

/* the following tables all added for v3.0 to support database migration from BIOTRAX */

/* physical cabinet */
CREATE TABLE CABINET (
    CABINET_ID  NUMBER(11,0) NOT NULL,
    NAME        VARCHAR2(50),
    COMMENTS    VARCHAR2(500),
    CONSTRAINT CABINET_PK PRIMARY KEY (CABINET_ID)
);

/* location of drawer in cabinet */
CREATE TABLE DRAWER (
    DRAWER_ID   NUMBER(11,0) NOT NULL,
    CABINET_ID  NUMBER(11,0) NOT NULL,
    DRAWER_NUMBER   NUMBER(5,0),                /* This might have incorrectly become DRAWER_FLOAT on some SQL Server systems due to a bug in the conversion script utility */
    CONSTRAINT DRAWER_PK PRIMARY KEY (DRAWER_ID),
    CONSTRAINT DRAWER_FK FOREIGN KEY (CABINET_ID) REFERENCES CABINET (CABINET_ID)
);

/* location of slide box in drawer */
CREATE TABLE SLIDEBOX (
    SLIDEBOX_ID     NUMBER(11,0) NOT NULL,
    DRAWER_ID       NUMBER(11,0),
    CELLTYPE        VARCHAR2(10),
    NAME            VARCHAR2(20),
    COLLECTION      VARCHAR2(1000),  /* to include collection name & authors */
    CONSTRAINT SLIDEBOX_PK PRIMARY KEY (SLIDEBOX_ID),
    CONSTRAINT SIDEBOX_FK FOREIGN KEY (DRAWER_ID) REFERENCES DRAWER (DRAWER_ID)
);

/* location of specimen cell within multi-cell slide box */
CREATE TABLE SLIDECELL (
    SLIDECELL_ID      NUMBER(11,0) NOT NULL,
    CELL_NUMBER       NUMBER(5,0),
    SLIDEBOX_ID       NUMBER(11,0) NOT NULL,
    CONSTRAINT SLIDECELL_PK PRIMARY KEY (SLIDECELL_ID),
    CONSTRAINT SLIDECELL_FK1 FOREIGN KEY (SLIDEBOX_ID) REFERENCES SLIDEBOX (SLIDEBOX_ID)
);

/* This is for an identified specimen from the taxonomic database. Similar in concept to a 
    type specimen, but for in-house collections
*/
CREATE TABLE SPECIMEN (
    SPECIMEN_ID     NUMBER(11,0) NOT NULL,
    SPEC_ID         NUMBER(11,0),
    SLIDECELL_ID    NUMBER(11,0),
    COMMENTS        VARCHAR2(1000),
    HIERACHY        VARCHAR2(50),
    AUTHOR          VARCHAR2(100),
    STUDY_SAMPLE_ID NUMBER(11,0),        /* For (obsolete) link to study_sample */
    CONSTRAINT SPECIMEN_PK PRIMARY KEY (SPECIMEN_ID),
    CONSTRAINT SPECIMEN_FK1 FOREIGN KEY (SPEC_ID) REFERENCES SPECIES (SPEC_ID)
);

CREATE TABLE PUBLICATION (
    PUBLICATION_ID  NUMBER(11,0) NOT NULL,
    TITLE           VARCHAR2(500),
    REFERENCE       VARCHAR2(250),
    PUB_INDEX       VARCHAR2(20),
    PUB_YEAR        NUMBER(5,0),
    AUTHOR          VARCHAR2(250),    
    CONSTRAINT  PUBLICATION_PK PRIMARY KEY (PUBLICATION_ID)
);

CREATE TABLE SPECIMEN_IMAGE (
    SPECIMEN_IMAGE_ID   NUMBER(11,0) NOT NULL,
    SPECIMEN_ID         NUMBER(11,0) NOT NULL,
    PICPATH             VARCHAR2(256),
    IMAGE               BLOB,
    IMAGE_THUMB         BLOB,
    ORIENTATION         VARCHAR2(50),
    PROCESSING          VARCHAR2(50),
    ORIGIN              VARCHAR2(30),
    COMMENTS            VARCHAR2(500),
    PUBLICATION_ID      NUMBER(11,0),
    CONSTRAINT SPECIMEN_IMAGE_PK PRIMARY KEY (SPECIMEN_IMAGE_ID),
    CONSTRAINT SPECIMEN_IMAGE_FK1 FOREIGN KEY (SPECIMEN_ID) REFERENCES SPECIMEN (SPECIMEN_ID),
    CONSTRAINT SPECIMEN_IMAGE_FK2 FOREIGN KEY (PUBLICATION_ID) REFERENCES PUBLICATION (PUBLICATION_ID)
);

/* The following tables added in 3.0.19 to support Chemostrat */

/* The base category to which a header value belongs.Similar concept to taxon categories */

CREATE TABLE CHEM_CAT (
    ID          NUMBER(5,0)     NOT NULL,
    NAME        VARCHAR2(100)   NOT NULL, 	/* The category name e.g. MO for Major Oxides */
    ABR         VARCHAR2(20)    NOT NULL, 	/* Abbreviation e.g. Mo */
    COLOUR      VARCHAR2(9),               /* Category colour */
    CONSTRAINT CHEM_CAT_PK PRIMARY KEY (ID)
);

/* The columns headers for each chemostrat element, compound or ratio, and determination of primary, derived internally or externally */
CREATE TABLE CHEM_HDR (
    ID          NUMBER(5,0)     NOT NULL,
    NAME        VARCHAR2(20)    NOT NULL, 	/* The common name e.g,. Al203 */
    TITLE       VARCHAR2(20),               /* The title using unicode chars or html to express subscript */
    UNITS       VARCHAR2(20)    NOT NULL,   /* one of PPM, %, g */
    TYPE        VARCHAR2(10)    NOT NULL,   /* Type of column - RAW, CALC_INTERNAL or CALC_EXTERNAL */
    FORMULA     VARCHAR2(100),              /* An expression using placeholders to derive curve from existing columns using their CHEM_ID. e.g. $1 / $2 would be a resut of the first parameter divided by the second. The parameters are given in the CHEM_HDR_FORMULA table */
    CHEM_CAT_ID NUMBER(5,0)     NOT NULL, 	/* Link to the category of this curve. Each CHEM_ID can only belong to one category */
    COLOUR VARCHAR2(9),
    CONSTRAINT CHEM_HDR_PK PRIMARY KEY (ID),
    CONSTRAINT CHEM_HDR_FK1 FOREIGN KEY (CHEM_CAT_ID) REFERENCES CHEM_CAT(ID)
);

/* Index table for placeholders in CHEM_HDR formula column. Ony used for derived curves. */
CREATE TABLE CHEM_HDR_FORMULA (
    CHEM_HDR_ID     NUMBER(5,0) NOT NULL, /* The ID of the curve derived through a formula */
    CHEM_FORMULA_ID NUMBER(5,0) NOT NULL, /* The CHEM_ID of the formula parameter */
    FORM_INDEX      NUMBER(5,0) NOT NULL, /* The index of the parameter in the formula */
    CONSTRAINT CHEM_HDR_FORMULA_PK PRIMARY KEY (CHEM_HDR_ID, CHEM_FORMULA_ID),
    CONSTRAINT CHEM_HDR_FORMULA_FK1 FOREIGN KEY (CHEM_HDR_ID) REFERENCES CHEM_HDR (ID),
    CONSTRAINT CHEM_HDR_FORMULA_FK2 FOREIGN KEY (CHEM_FORMULA_ID) REFERENCES CHEM_HDR (ID)
);

/* Represents a suite of chemostrat analyses. Multiple suites can be defined so there are can be more than one suite per well if there are separate anlyses. */
CREATE TABLE CHEM_ANALY_HDR (
    ID              NUMBER(11,0)    NOT NULL,   /* Unique ID for this set of analyses (suite), there can be more than one suite for a well */
    WELL_ID         NUMBER(11,0)    NOT NULL,   /* The well that this suite belongs to */
    ANALY_NO        NUMBER(5,0)     NOT NULL,   /* The number of the suite of analyses, unique and starting at 1 for the first suite. */
    SOURCE VARCHAR2(100),                       /* Name of the data souce or lab providing the sample analyses */
    COMMENTS VARCHAR2(1000),
    CREATED         DATE,
    CREATOR         NUMBER(5,0),
    MODIFIED        DATE,
    MODIFIER        NUMBER(5,0),
    UPDATED         TIMESTAMP,
    UPDATER         NUMBER(5,0),
    CONSTRAINT CHEM_ANALY_HDR_PK PRIMARY KEY (ID),
    CONSTRAINT CHEM_ANALY_HDR_FK1 FOREIGN KEY (WELL_ID) REFERENCES WELLS(WELL_ID),
    CONSTRAINT CHEM_ANALY_HDR_FK2 FOREIGN KEY (CREATOR) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHEM_ANALY_HDR_FK3 FOREIGN KEY (MODIFIER) REFERENCES USERDEF(USER_ID),
    CONSTRAINT CHEM_ANALY_HDR_FK4 FOREIGN KEY (UPDATER) REFERENCES USERDEF(USER_ID),
);    

/* The definition of a sample, analysed for chemostrat */
CREATE TABLE CHEM_ANALYSIS (
    ID                  NUMBER(11,0) NOT NULL,
    CHEM_ANALY_HDR_ID   NUMBER(11,0) NOT NULL,
    WELL_ID             NUMBER(11,0) NOT NULL,
    SAMP_ID             NUMBER(11,0) NOT NULL, /* Foreign key to SAMPLES table */
    CONSTRAINT CHEM_ANALYSIS_PK PRIMARY KEY (ID),
    CONSTRAINT CHEM_ANALYSIS_FK1 FOREIGN KEY (CHEM_ANALY_HDR_ID) REFERENCES CHEM_ANALY_HDR (ID),
    CONSTRAINT CHEM_ANALYSIS_FK2 FOREIGN KEY (WELL_ID,SAMP_ID) REFERENCES SAMPLES (WELL_ID,SAMP_ID)
);

/* The chemostrat data values, per analysis */
CREATE TABLE CHEM_TRACE (
    CHEM_ANALYSIS_ID    NUMBER(11,0)    NOT NULL,
    CHEM_HDR_ID         NUMBER(5,0)     NOT NULL,
    CHEM_VALUE          NUMBER(12,8),   /* The value of the curve, in the units defined by the CHEM_HDR table */
    CONSTRAINT CHEM_TRACE_PK PRIMARY KEY (CHEM_ANALYSIS_ID, CHEM_HDR_ID),
    CONSTRAINT CHEM_TRACE_FK1 FOREIGN KEY (CHEM_ANALYSIS_ID) REFERENCES CHEM_ANALYSIS (ID),
    CONSTRAINT CHEM_TRACE_FK2 FOREIGN KEY (CHEM_HDR_ID) REFERENCES CHEM_HDR (ID)
);

/* A group of curves, artitrary, user generated';/*A group of curves, artitrary, user generated */
CREATE TABLE CHEM_GROUP (
    ID              NUMBER(5,0) NOT NULL,
    NAME            VARCHAR2(80) NOT NULL,
    CONSTRAINT CHEM_GROUP_PK PRIMARY KEY (ID)
);

/* Membership of chem group */
CREATE TABLE CHEM_GROUP_MBR (
    CHEM_GROUP_ID   NUMBER(5,0) NOT NULL,
    CHEM_HDR_ID     NUMBER(5,0) NOT NULL,
    COLOUR          VARCHAR2(9),
    CONSTRAINT CHEM_GROUP_MBR_PK1 PRIMARY KEY (CHEM_GROUP_ID, CHEM_HDR_ID),
    CONSTRAINT CHEM_GROUP_MBR_FK1 FOREIGN KEY (CHEM_GROUP_ID) REFERENCES CHEM_GROUP (ID),
    CONSTRAINT CHEM_GROUP_MBR_FK2 FOREIGN KEY (CHEM_HDR_ID) REFERENCES CHEM_HDR (ID)
);

/* The properties of the chem curve(s) in a chart panel. The panel_no gives scope for multiple curves in one panel */
CREATE TABLE CHT_CHEM (
    PANEL_ID            NUMBER(11,0) NOT NULL,
    PANEL_NO            NUMBER(5,0) NOT NULL,
    CHEM_HDR_ID         NUMBER(5,0),
    CHEM_GROUP_ID       NUMBER(5,0),
    CHEM_CAT_ID         NUMBER(5,0),
    PROP                VARCHAR2(255),
    CONSTRAINT CHT_CHEM_PK PRIMARY KEY (PANEL_ID, PANEL_NO),
    CONSTRAINT CHT_CHEM_FK1 FOREIGN KEY (CHEM_HDR_ID) REFERENCES CHEM_HDR (ID),
    CONSTRAINT CHT_CHEM_FK2 FOREIGN KEY (CHEM_GROUP_ID) REFERENCES CHEM_GROUP (ID),
    CONSTRAINT CHT_CHEM_FK3 FOREIGN KEY (CHEM_CAT_ID) REFERENCES CHEM_CAT (ID),
    CONSTRAINT CHT_CHEM_FK4 FOREIGN KEY (PANEL_ID) REFERENCES CHTPANL (PANEL_ID)
);

INSERT INTO DBVERSION VALUES ('3.0', 'MapBlock support', sysdate);
COMMIT;
